//	CalendarWindow.cpp
//
//	Tetsuo Yamada
//	Created: Mon, Feb. 08 1999
//	Modified: Mon, Dec. 06 1999


#include	<string.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<math.h>
#include	<interface/Box.h>
#include	<interface/Screen.h>
#include	<interface/ScrollView.h>
#include	"CalendarWindow.h"
#include	"DateView.h"



const char *d_last_month = "January 2038" ;
const char *d_first_month = "January 1970" ;



//==================================================================
//	global functions which helps struct tm
//==================================================================
int32 days_in_month (const struct tm *t) {
	switch (t->tm_mon) {
		case 0 : //	Jan
		case 2 : //	Mar
		case 4 : //	May
		case 6 : //	Jul
		case 7 : //	Aug
		case 9 : //	Oct
		case 11 : //	Dec
			return 31 ;
		case 3 : //	Apr
		case 5 : //	Jun
		case 8 : //	Sep
		case 10 : //	Nov
			return 30 ;
		case 1 : //	Feb
			return (((t->tm_year + 1900)%4 == 0 && (t->tm_year + 1900)%100 != 0 || 
					(t->tm_year + 1900)%400 == 0) ? 29 : 28) ;
	}
	
	return -1 ;
} ;

int first_weekday (const struct tm *t) {
	int weekday = t->tm_wday ;
//	int day = t->tm_mday ;
	
	weekday -= (t->tm_mday - 1) ;
	weekday %= 7 ;
	if (weekday < 0) weekday += 7 ;
	return weekday ;
} ;

struct tm *now_as_tm () {
	time_t t = time (NULL) ;
	return localtime (&t) ;
} ;


//==================================================================
//	class CalendarView
//==================================================================
CalendarView::CalendarView (BPoint lefttop, const char *name, 
				const struct tm *when, uint32 resize) 
			: BView (BRect (lefttop.x, lefttop.y, lefttop.x + 10, lefttop.y + 10), 
				name, resize, B_WILL_DRAW) {
	fWhen = when ;
} ;

CalendarView::~CalendarView () {
	//	do nothing
} ;

void CalendarView::AttachedToWindow () {
	if (Parent()) {
		SetViewColor (Parent()->ViewColor()) ;
		SetLowColor (Parent()->LowColor()) ;
		SetHighColor (Parent()->HighColor()) ;
	}
	
	ResizeToPreferred () ;
	
	BView::AttachedToWindow () ;
} ;

void CalendarView::GetPreferredSize (float *width, float *height) {
	BFont font ;
	float blockWidth = 0 ;
	float blockHeight = 0 ;
	
	GetFont (&font) ;
	
	blockWidth = font.StringWidth ("30") + 6 ;
	blockHeight = font.Size() + 4 ;
	*width = blockWidth*7 ;
	*height = blockHeight*7 ;
} ;

void CalendarView::Draw (BRect) {
//	BRect bounds = Bounds() ;
	BFont font ;
	BRect block ;
	int32 numDays = days_in_month (fWhen) ;
	int weekday = first_weekday (fWhen) ;
	int weekIndex = 1 ;
	struct tm *today = now_as_tm () ;
	char string [8] ;
//	float blockWidth = 0, blockHeight = 0 ;
	
	static const char *wds[7] = {"S", "M", "T", "W", "T", "F", "S"} ;
	
	GetFont (&font) ;
	block.left = block.top = 0 ;
	block.right = font.StringWidth ("30") + 6 - 1 ;
	block.bottom = font.Size() + 4 - 1 ;
	
	//	draw weekdays
	for (int32 i = 0 ; i < 7 ; i++) {
		switch (i) {
			case 0 : 
				SetHighColor (*d_holiday) ;
				break ;
			case 6 : 
				SetHighColor (*d_saturday) ;
				break ;
			default : 
				SetHighColor (*d_weekday) ;
				break ;
		}
		DrawAlignedString (wds[i], block) ;
		block.OffsetBy (block.Width() + 1, 0) ;
	}
	
	//	draw days
	block.OffsetTo (weekday*(block.Width() + 1), block.Height() + 1) ;
	for (int32 day = 1 ; day <= numDays ; day++) {
		sprintf (string, "%ld", day) ;
		
		switch ((day + weekday + 6)%7) {
			case 0 : 
				SetHighColor (*d_holiday) ;
				break ;
			case 6 : 
				SetHighColor (*d_saturday) ;
				break ;
			default : 
				SetHighColor (*d_weekday) ;
				break ;
		}
		//	Precisely, I must consider holidays.
		
		if ((fWhen->tm_year == today->tm_year) && 
			(fWhen->tm_mon == today->tm_mon) && 
			(day == today->tm_mday)) {
			//	wow. it's today.
			DrawAlignedString (string, block, true) ;
		} else {
			DrawAlignedString (string, block) ;
		}
		
		switch ((day + weekday + 6)%7) {
			case 6 : 
				block.OffsetBy (-6*(block.Width() + 1), block.Height() + 1) ;
				weekIndex++ ;
				break ;
			default : 
				block.OffsetBy (block.Width() + 1, 0) ;
				break ;
		}
	}
} ;

void CalendarView::DrawAlignedString (const char *string, BRect block, bool emphasize) {
	float width = StringWidth (string) ;
	font_height fh ;
	BPoint p ;
	
	//	calculate the point to draw
	GetFontHeight (&fh) ;
	p.x = block.right - width - 1 ;
	p.y = block.top + floor (block.Height()/2 + fh.ascent/2) ;
	
	if (emphasize) {
		rgb_color low = LowColor() ;
		
		SetLowColor (tint_color (low, B_DARKEN_1_TINT)) ;
		block.left = p.x - 2 ;
		FillRect (block, B_SOLID_LOW) ;
		DrawString (string, p) ;
		
		SetLowColor (low) ;
	} else {
		DrawString (string, p) ;
	}
} ;

void CalendarView::MouseUp (BPoint where) {
	Window()->PostMessage (B_QUIT_REQUESTED) ;
	BView::MouseUp (where) ;
} ;


//==================================================================
//	class IteratorButton
//==================================================================
IteratorButton::IteratorButton (BRect frame, const char *name, 
				iterator_type type, BMessage *msg, uint32 resize) 
			: BControl (frame, name, B_EMPTY_STRING, msg, resize, 
				B_WILL_DRAW/*|B_NAVIGABLE*/|B_FRAME_EVENTS) {
	fLast = 0 ;
	fCounter = 0 ;
	fType = type ;
} ;

IteratorButton::~IteratorButton () {
} ;

void IteratorButton::Draw (BRect) {
	const rgb_color origLow = ViewColor() ;
	const rgb_color origHigh = HighColor() ;
	rgb_color low ;
	/*rgb_color high ;*/
	BPoint center ;
	int th = 0 ; // triangle height
	BPoint p1, p2, p3 ;
	
	//	fill bounds with the correct color.
	if (Value()==B_CONTROL_ON) low = tint_color (origLow, B_DARKEN_1_TINT) ;
	else low = origLow ;
	SetLowColor (low) ;
	FillRect (fBounds, B_SOLID_LOW) ;
	
	//	make it 3D
	if (Value()==B_CONTROL_ON) SetLowColor (tint_color (low, B_LIGHTEN_2_TINT)) ;
	else SetLowColor (tint_color (low, B_DARKEN_1_TINT)) ;
	StrokeLine (fBounds.RightTop(), fBounds.RightBottom(), B_SOLID_LOW) ;
	StrokeLine (fBounds.LeftBottom(), B_SOLID_LOW) ;
	if (Value()==B_CONTROL_ON) SetLowColor (tint_color (low, B_DARKEN_1_TINT)) ;
	else SetLowColor (tint_color (low, B_LIGHTEN_2_TINT)) ;
	StrokeLine (fBounds.LeftTop(), B_SOLID_LOW) ;
	StrokeLine (fBounds.RightTop(), B_SOLID_LOW) ;
	SetLowColor (low) ;
	
	//	get triangle
	center.Set (fBounds.Width()/2, floor (fBounds.Height()/2 + 0.5)) ;
	th = (int)(floor (fBounds.Width() + 0.5)) - 4 ;
	while (th >= fBounds.Height()/2 + 0.5 - 2) th -= 2 ;
	
	p1.y = center.y ;
	p2.y = center.y - th ;
	p3.y = center.y + th ;
	if (fType==D_NEXT) {
		p1.x = center.x + (float)th/2 ;
		p2.x = p3.x = center.x - (float)th/2 ;
	} else {
		p1.x = center.x - (float)th/2 ;
		p2.x = p3.x = center.x + (float)th/2 ;
	}
	
	//	draw triangle
	if (!IsEnabled()) SetHighColor (tint_color (origHigh, B_DISABLED_MARK_TINT)) ;
	FillTriangle (p1, p2, p3, B_SOLID_HIGH) ;
	
	//	sweep
	SetLowColor (origLow) ;
	SetHighColor (origHigh) ;
} ;

status_t IteratorButton::Invoke (BMessage *msg) {
	if (!msg) msg = Message() ;
	if (!msg) return BControl::Invoke (msg) ;
	
	msg->AddInt32 ("iterator_type", (int32)fType) ;
	return BControl::Invoke (msg) ;
} ;

void IteratorButton::AllAttached () {
	fBounds = Bounds() ;
	BControl::AllAttached () ;
} ;

void IteratorButton::FrameResized (float width, float height) {
	fBounds = Bounds() ;
	BControl::FrameResized (width, height) ;
	Invalidate () ;
} ;

void IteratorButton::Keizoku () {
	BPoint p ;
	uint32 buttons = 0 ;
	bigtime_t interval = 0 ;
	
	GetMouse (&p, &buttons) ;
	
	if (buttons && fBounds.Contains (p) && IsEnabled() && Value()==B_CONTROL_ON) {
		//	KEIZOKU
		Looper()->PostMessage (D_KEIZOKU, this) ;
		
		switch (fCounter) {
			case 0 : 
				interval = 0 ;
				break ;
			case 1 : 
			case 2 : 
				interval = 400*1000 ;
				break ;
			case 3 : 
			case 4 : 
				interval = 300*1000 ;
				break ;
			case 5 : 
			case 6 : 
			case 7 : 
				interval = 200*1000 ;
				break ;
			case 8 : 
			case 9 : 
			case 10 : 
			case 11 : 
			case 12 : 
				interval = 150*1000 ;
				break ;
			default : 
				interval = 100*1000 ;
				break ;
		}
		
		if (system_time() - fLast > interval) {
			Invoke () ;
			fLast = system_time() ;
			fCounter++ ;
		}
	} else {
		SetValue (B_CONTROL_OFF) ;
		fLast = 0 ;
		fCounter = 0 ;
	}
} ;

void IteratorButton::MouseDown (BPoint p) {
	p = p ;
	if (IsEnabled()) {
		SetValue (B_CONTROL_ON) ;
		Keizoku () ;
	}
} ;

void IteratorButton::MessageReceived (BMessage *msg) {
	switch (msg->what) {
		case D_KEIZOKU : 
			Keizoku () ;
			break ;
		default : 
			BControl::MessageReceived (msg) ;
			break ;
	}
} ;


//==================================================================
//	class CalendarWindow
//==================================================================
void CalendarWindow::BuildChildren () {
	const float space = 2 ;
	const float buttonWidth = 9 ;
	const float buttonHeight = 17 ;
	BBox *bottom = NULL, *title = NULL ;
	BScrollView *scr = NULL ;
	IteratorButton *next = NULL, *prev = NULL ;
	BRect bounds = Bounds() ;
	BRect rect ;
	char label[128] ;
	
	bottom = new BBox (bounds, "bottom", B_FOLLOW_ALL, B_WILL_DRAW|B_FRAME_EVENTS, 
			B_PLAIN_BORDER) ;
	AddChild (bottom) ;
	bottom->SetViewColor (ui_color (B_MENU_BACKGROUND_COLOR)) ;
	bottom->SetLowColor (bottom->ViewColor()) ;
	
	//	build title. 
	rect = bounds ;
	rect.InsetBy (space + 2, space + 2) ;
	rect.bottom = rect.top + buttonHeight - 1 ;
	title = new BBox (rect, "title", B_FOLLOW_LEFT_RIGHT|B_FOLLOW_TOP, 
			B_WILL_DRAW, B_NO_BORDER) ;
	
	rect.OffsetTo (0, 0) ;
	rect.left = rect.right - (buttonWidth - 1) ;
	next = new IteratorButton (rect, "next_", D_NEXT, new BMessage (D_NEXT_MONTH)) ;
	title->AddChild (next) ;
	next->SetTarget (this) ;
	
	rect.OffsetBy (-buttonWidth, 0) ;
	prev = new IteratorButton (rect, "prev_", D_PREV, new BMessage (D_PREV_MONTH)) ;
	title->AddChild (prev, next) ;
	prev->SetTarget (this) ;
	
	rect.right = rect.left - 1 ;
	rect.left = 0 ;
	GetMonthString (label, 127) ;
	fMonth = new BStringView (rect, "month", label, B_FOLLOW_LEFT_RIGHT|B_FOLLOW_TOP) ;
	title->AddChild (fMonth) ;
	
	scr = new BScrollView ("scr", title, B_FOLLOW_LEFT_RIGHT|B_FOLLOW_TOP) ;
	bottom->AddChild (scr) ;
	scr->SetViewColor (bottom->ViewColor()) ;
	scr->SetLowColor (bottom->ViewColor()) ;
	
	next->SetViewColor (bottom->ViewColor()) ;
	prev->SetViewColor (bottom->ViewColor()) ;
	
	//	add calendar view
	rect = scr->Frame() ;
	fCalendar = new CalendarView (BPoint (rect.left, rect.bottom + space + 4), 
			"calendar", &fWhen) ;
	bottom->AddChild (fCalendar) ;
	fCalendar->SetViewColor (bottom->ViewColor()) ;
	fCalendar->SetLowColor (bottom->LowColor()) ;
	fCalendar->ResizeToPreferred () ;
	
	//	resize window
	rect = fCalendar->Frame() ;
	ResizeTo (rect.right + space, rect.bottom + space) ;
} ;

void CalendarWindow::GetMonthString (char *buf, size_t size) const {
	size_t s = strftime (buf, size, "%B %Y", &fWhen) ;
	
	if (s==0) strcpy (buf, "Failed to get Month") ;
} ;

BPoint CalendarWindow::ScreenLocation (BPoint where) {
	const BPoint delta (4, 4) ;
	BRect screen = BScreen (this).Frame() ;
	BRect frame = Bounds() ;
	BPoint result = where ;
	
	result -= delta ;
	if (result.x < screen.left) 
		result.x = screen.left - result.x ;
	else if (result.x + frame.Width() > screen.right) 
		result.x = screen.right - frame.Width() ;
	if (result.y < screen.top) 
		result.y = screen.top - result.y ;
	else if (result.y + frame.Height() > screen.bottom) 
		result.y = screen.bottom - frame.Height() ;
	
	return result ;
} ;

/*BPoint CalendarWindow::ScreenLocation (BPoint point) {
	const int8 delta = 2 ;
	BPoint result (0, 0) ;
	BRect screen = BScreen (this).Frame() ;
	BRect bounds = Bounds() ;
	
	if (bounds.Width() + point.x > screen.right) 
		result.x = point.x - bounds.Width() + delta ;
	else 
		result.x = point.x - delta ;
	if (bounds.Height() + point.y > screen.bottom) 
		result.y = point.y - bounds.Height() + delta ;
	else 
		result.y = point.y - delta ;
	
	return result ;
} ;*/

CalendarWindow::CalendarWindow (BPoint p) 
			: BWindow (BRect (p.x, p.y, p.x + 100, p.y + 100), "Calendar", 
				B_BORDERED_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL, 
				B_NOT_MOVABLE|B_NOT_RESIZABLE|B_NOT_ZOOMABLE|B_NOT_MINIMIZABLE) {
	fCalendar = NULL ;
	fMonth = NULL ;
	
	time_t t = time (NULL) ;
	struct tm *ptr = localtime (&t) ;
	
	fWhen = *ptr ;
	
	BuildChildren () ;
	MoveTo (ScreenLocation (p)) ;
} ;

CalendarWindow::~CalendarWindow () {
} ;

void CalendarWindow::WindowActivated (bool active) {
	if (!active) {
		PostMessage (B_QUIT_REQUESTED) ;
	}
	
	BWindow::WindowActivated (active) ;
} ;

void CalendarWindow::MessageReceived (BMessage *msg) {
	char buf [128] ;
	IteratorButton *prev = dynamic_cast<IteratorButton*>(FindView("prev_")) ;
	IteratorButton *next = dynamic_cast<IteratorButton*>(FindView("next_")) ;
	
	switch (msg->what) {
		case D_NEXT_MONTH : 
			fWhen.tm_mon++ ;
			mktime (&fWhen) ;
			GetMonthString (buf, 127) ;
			fMonth->SetText (buf) ;
			fCalendar->Invalidate () ;
			
			if (strcmp (buf, d_last_month) == 0) {
				next->SetValue (0) ;
				next->SetEnabled (false) ;
			}
			if (prev->IsEnabled() == false) prev->SetEnabled (true) ;
			break ;
		
		case D_PREV_MONTH : 
			fWhen.tm_mon-- ;
			mktime (&fWhen) ;
			GetMonthString (buf, 127) ;
			fMonth->SetText (buf) ;
			fCalendar->Invalidate () ;
			
			if (strcmp (buf, d_first_month) == 0) {
				prev->SetValue (0) ;
				prev->SetEnabled (false) ;
			}
			if (next->IsEnabled() == false) next->SetEnabled (true) ;
			break ;
			
		default : 
			BWindow::MessageReceived (msg) ;
			break ;
	}
} ;

void CalendarWindow::Show () {
	snooze (150*1000) ;
	BWindow::Show () ;
} ;
