//*****************************************************************************
//
//	File:		 GfxWindow.cpp
//
//	Description: WindowScreen class for BeBench.
//
//  bases on Domino from Be
//
//	Copyright 1997, Dirk Steins
//
//*****************************************************************************


#define DEBUG 1
#include <stdlib.h>
#include <support/Debug.h>


#include "GfxWindow.h"
#include "BenchGlobal.h"
#include "colors.h"


//*********************************************************************

#define MAX_COLS		240		/* colors in color list. */ 

#define BLIT_X 303 // not giving any special treatment to even numbered bitblits...
#define BLIT_Y 303

// Index of the blit graphics hook.
#define	INDEX_LINE8		3
#define	INDEX_LINE32	4
#define	INDEX_RECT8		5
#define	INDEX_RECT32	6
#define	INDEX_BLIT		7
#define	INDEX_ARRAY8	8
#define	INDEX_ARRAY32	9

#pragma mark --Variables--

int      drawing_enabled = FALSE;
int      kill_play = FALSE;
//int      page_num;
//sem_id   lock;
//sem_id   ender;
//GfxWindow *window;
//long     row;
//BRect	 gBounds;
//
//uchar    *draw_base;

// These are only reference by the GfxWindow
static int       First = TRUE;




#pragma mark --GfxWindow Class--
//*********************************************************************
// WindowScreen specific stuff.

GfxWindow::GfxWindow (long bench, BWindow* parent, long space, char *name):BWindowScreen(name,space)
{
	BRect   rect;
	
	mDepth = (space==B_8_BIT_640x480)?8:32;
	mBench = bench;
	mParent = parent;

	// allocate color list
	color_list = (rgb_color*) malloc (sizeof (rgb_color) * MAX_COLS);

	// initialize color cycling table
	int idx = 0;
	int rgbTab[3] = {255, 0, 0};
	int rgbIdx = 0;
	int rgbRunTab[6] = {1, 0, 2, 1, 0, 2};
	int step = 255 / 6;
	for (idx = 0; idx < MAX_COLS; idx++)
	{
		color_list[idx].red = (uchar) rgbTab[0];
		color_list[idx].green =  (uchar) rgbTab[1];
		color_list[idx].blue = (uchar) rgbTab[2];
		color_list[idx].alpha =  (uchar) 0;
		
		rgbTab[rgbRunTab[rgbIdx]] += step;
		if (rgbTab[rgbRunTab[rgbIdx]] > 255)
		{
			rgbTab[rgbRunTab[rgbIdx]] = 255;
			step = -step;
			rgbIdx++;
		}
		else if (rgbTab[rgbRunTab[rgbIdx]] < 0)
		{
			rgbTab[rgbRunTab[rgbIdx]] = 0;
			step = -step;
			rgbIdx++;
		}
		if (rgbIdx > 5) rgbIdx = 0;
	}

// create the semaphore
//	lock = create_sem(1,"GfxWindow lock sem"); 
//	ender = create_sem(0,"GfxWindow kill sem");

// Initialize Controls
	First = TRUE;
	kill_play = FALSE;
	drawing_enabled = FALSE;

// spawn the domino player thread
//	draw = spawn_thread(BenchRunner,"GfxWindow bench",B_NORMAL_PRIORITY,(void*)this);
//	resume_thread(draw);
}


void GfxWindow::Quit()
{
	Disconnect();

// If the domino player is running, ask for its death and wait for it...
	if (draw != 0L) 
	{
		kill_play = TRUE;
//		acquire_sem(ender);
	}
// free the semaphores
//	delete_sem(ender);
//	delete_sem(lock);
// Quit the WindowScreen.
	BWindowScreen::Quit();
}

void GfxWindow::ScreenConnected(bool active)
{
	short    height;

//	acquire_sem(lock);
	if (active == TRUE) {
//		if (First) {
//			if (ProposeFrameBuffer(mDepth, 640, &height) == B_NO_ERROR)
//			{
//				if (height >= 480)
//					if (SetFrameBuffer(480, 640, 480) == B_NO_ERROR)
//						goto ok;
//			}
//			PostMessage(B_QUIT_REQUESTED);
//			goto end;
//		ok:
//			First = FALSE;
//		}
//		MoveDisplayArea(0, 0); // set page 0
	
		// get graphic card hooks
		blit_jmp = (BLIT)CardHookAt(INDEX_BLIT);
		line_8 = (LINE8)CardHookAt (INDEX_LINE8);
		line_32 = (LINE32)CardHookAt (INDEX_LINE32);
		rect_8 = (RECT8)CardHookAt (INDEX_RECT8);
		rect_32 = (RECT32)CardHookAt (INDEX_RECT32);
		line_array_8 = (ARRAY8)CardHookAt (INDEX_ARRAY8);
		line_array_32 = (ARRAY32)CardHookAt (INDEX_ARRAY32);
		
		if (blit_jmp == NULL)
		{
			PRINT (("Gfx: blit not supported\n")); 
		}
		if (line_8 == NULL)
		{
			PRINT (("Gfx: 8bit line not supported\n")); 
		}
		if (line_32 == NULL)
		{
			PRINT (("Gfx: 32 bit line not supported\n")); 
		}
		if (rect_8 == NULL)
		{
			PRINT (("Gfx: 8 bit rect not supported\n")); 
		}
		if (rect_32 == NULL)
		{
			PRINT (("Gfx: 32 bit rect not supported\n")); 
		}
		if (line_array_8 == NULL)
		{
			PRINT (("Gfx: 8 bit line array not supported\n")); 
		}
		if (line_array_32 == NULL)
		{
			PRINT (("Gfx: 32 bit line array not supported\n")); 
		}

		bounds = BRect (0,0,639,479);
		// allow the drawing in the frame buffer (when lock release).
		drawing_enabled = TRUE;

		snooze (1e6);
		BenchRunner ((void*) this);
		
	}
	else
	// stop drawing.
		drawing_enabled = FALSE;
 end:
// Release the main semaphore.
//	release_sem(lock);
	;
}

#pragma mark --Drawing--

// these functions are inline so they are actually located
//	in the header file GfxWindow.h

/*inline void plot_point(long x, long y,uchar inColor)
{
	if ( gBounds.left <= x && gBounds.right > x+1 && gBounds.top <= y && gBounds.bottom > y+1 )
	{
		*(draw_base + (y * row)+ x ) = inColor;
	}
}

inline void plot_dot(long x, long y,uchar inColor)
{
	uchar *theAdd;
	if ( gBounds.left <= x && gBounds.right > x+2 && gBounds.top <= y && gBounds.bottom > y+2 )
	{
		theAdd = (draw_base + (y * row)+ x );
		*theAdd = inColor;
		*(theAdd+row) = inColor;
		*(theAdd+1) = inColor;
		*(theAdd+row+1) = inColor;
		
	}
}	*/

long GfxWindow::BenchRunner(void	*data)
{
	float result;
	GfxWindow*	thisWdw;
	int		benchList[8] = 
			{
			X_BEN_WBLITSS,
			X_BEN_WFILLRC,
			X_BEN_WVLINES,
			X_BEN_WHLINES,
			X_BEN_WDLINES,
			X_BEN_WVLINAR,
			X_BEN_WHLINAR,
			X_BEN_WDLINAR
			};

	thisWdw = (GfxWindow*) data;
	
	if (thisWdw->mBench == X_BEN_RUNALL)
	{
		for (int i = 0; i < 8; i++)
		{
			thisWdw->RunBench (benchList[i], result);
			
			// Post result
			BMessage* msg = new BMessage;
			
			msg->what = X_BEN_RESULT;
			msg->AddLong ("bench", benchList[i]);
			msg->AddFloat ("bres", result);
			
			thisWdw->mParent->PostMessage (msg);
						
			snooze (5e5);
		}
	}
	else
	{
		thisWdw->RunBench (thisWdw->mBench, result);

		// Post result
		BMessage* msg = new BMessage;
		
		msg->what = X_BEN_RESULT;
		msg->AddLong ("bench", thisWdw->mBench);
		msg->AddFloat ("bres", result);
		
		thisWdw->mParent->PostMessage (msg);
		
	}
	
	thisWdw->PostMessage (B_QUIT_REQUESTED);
}

void GfxWindow::RunBench(int bench, float& result)
{
	switch (bench)
	{
		case X_BEN_WBLITSS:
			result = RunBitBlitSS();
			break;
		case X_BEN_WFILLRC:
			result = RunFillRect();
			break;
		case X_BEN_WVLINES:
			result = RunVlines();
			break;
		case X_BEN_WHLINES:
			result = RunHlines();
			break;
		case X_BEN_WDLINES:
			result = RunDlines();
			break;
		case X_BEN_WVLINAR:
			result = RunVlineArray();
			break;
		case X_BEN_WHLINAR:
			result = RunHlineArray();
			break;
		case X_BEN_WDLINAR:
			result = RunDlineArray();
			break;
		default:
			break;
	}
}

// returns true if test was long enough, 
// otherwise false and calculates new run number
bool GfxWindow::CheckRunTime (double test_time, long &runs, long &runs_10)
{
	if (test_time < MIN_GFX_TIME)   /* if total time in test < min time */
	{
		if ((test_time) < MIN_MEASURE)  /* if total time in test < 0.1 */
		{
			runs = MIN_GFX_TIME/MIN_MEASURE* runs; /* increase number of runs by factor 100 */
			runs_10 = runs/240;
		}
		else                         /* if 10 > total time in test > 0.1 */
		{
			runs = MIN_GFX_TIME*MARGINAL/(test_time)* runs; /* increase runs by enough to make tottime > 10 */
			runs_10 = runs/240;
		}
	}
	else                              /* if total time in test > 10 */
	{
		return true;                          /* stop test */
	}                               /* otherwise repeat with new run # */
	return false;
}

// Graphics primitives
// for drawing patterns

void GfxWindow::SetHighColor (rgb_color highColor)
{
	mHighColor = highColor;
}

void GfxWindow::FillRect8 (BRect rect)
{
	if (rect_8 != NULL)
	{
		(rect_8)((long) rect.left, (long) rect.top, (long) rect.right, (long) rect.bottom,
			(uchar) index_for_color (mHighColor));
	}
}

void GfxWindow::FillRect32 (BRect rect)
{
	ulong color;
	
	color = (mHighColor.red << 24) + (mHighColor.green << 16) + (mHighColor.blue << 8) + (mHighColor.alpha);
	if (rect_32 != NULL)
	{
		(rect_32)((long) rect.left, (long) rect.top, (long) rect.right, (long) rect.bottom,
			color);
	}
}

void GfxWindow::FillRect (BRect rect)
{
	if (mDepth == 8)
	{
		FillRect8 (rect);
	}
	else if (mDepth == 32)
	{
		FillRect32 (rect);
	}
}

void GfxWindow::StrokeLine8 (BPoint from, BPoint to)
{
	if (line_8 != NULL)
	{
		(line_8)((long)from.x, (long) to.x, (long) from.y, (long) to.y,
			  (uchar) index_for_color (mHighColor), true, 
			  (short) bounds.left, (short) bounds.top, 
			  (short) bounds.right, (short) bounds.bottom);
	}
}

void GfxWindow::StrokeLine32 (BPoint from, BPoint to)
{
	ulong color;
	
//	PRINT (("StrokeLine32\n from (%d,%d) to (%d,%d)\n", (long)from.x, (long)from.y, (long) to.x, (long) to.y));
	color = (mHighColor.red << 24) + (mHighColor.green << 16) + (mHighColor.blue << 8) + (mHighColor.alpha);
	if (line_32 != NULL)
	{
		(line_32)((long)from.x, (long) to.x, (long) from.y, (long) to.y,
			  color, true, 
			  (short) bounds.left, (short) bounds.top, 
			  (short) bounds.right, (short) bounds.bottom);
	}
}

void GfxWindow::StrokeLine (BPoint from, BPoint to)
{
	if (mDepth == 8)
	{
		StrokeLine8 (from, to);
	}
	else if (mDepth == 32)
	{
		StrokeLine32 (from, to);
	}
}

void GfxWindow::BeginLineArray (long count)
{
//	PRINT (("BeginLineArray (%d)\n", count));
	if (mDepth == 8)
	{
		mIdxColArray = (indexed_color_line*) malloc (count * sizeof (indexed_color_line));
		mIdxNum = count;
	}
	else if (mDepth == 32)
	{
		mColArray = (rgb_color_line*) malloc (count * sizeof (rgb_color_line));
		mIdxNum = count;
	}
	mLaIdx = 0;
}

void GfxWindow::AddLine (BPoint from, BPoint to, rgb_color color)
{
//	PRINT (("AddLine %d\n", mLaIdx));
	if (mDepth == 8)
	{
		uchar colIdx = index_for_color (color);
		
		mIdxColArray[mLaIdx].x1 = (short) from.x;
		mIdxColArray[mLaIdx].y1 = (short) from.y;
		mIdxColArray[mLaIdx].x2 = (short) to.x;
		mIdxColArray[mLaIdx].y2 = (short) to.y;
		mIdxColArray[mLaIdx++].color = colIdx;
	}
	else if (mDepth == 32)
	{
		mColArray[mLaIdx].x1 = (short) from.x;
		mColArray[mLaIdx].y1 = (short) from.y;
		mColArray[mLaIdx].x2 = (short) to.x;
		mColArray[mLaIdx].y2 = (short) to.y;
		mColArray[mLaIdx++].color = color;
	}
} 

void GfxWindow::EndLineArray ()
{
//	PRINT (("EndLineArray (%d)\n", mIdxNum));
	if (mDepth ==  8)
	{
		if (line_array_8 != NULL)
		{
			(line_array_8) (mIdxColArray, mIdxNum, false, 
			(short) bounds.left, (short) bounds.top, (short) bounds.right, 
			(short) bounds.bottom);
		}
		free (mIdxColArray);
	}
	else if (mDepth == 32)
	{
		if (line_array_32 != NULL)
		{
			(line_array_32) (mColArray, mIdxNum, false, 
			(short) bounds.left, (short) bounds.top, 
			(short) bounds.right, (short) bounds.bottom);
		}
		free (mColArray);
	}
}

void GfxWindow::CopyBits (BRect source, BRect dest)
{
	if (blit_jmp != NULL) 
	{
		(blit_jmp)((long)source.left, (long)source.top, (long)dest.left, (long) dest.top,
			   (long) source.Width(), (long) source.Height());
	}
}

void GfxWindow::Sync ()
{
	if (sync != NULL) 
	{
		(sync)();
	}
}

// Benchmark routines.
// same as GfxBench routines

float GfxWindow::RunBitBlitSS()
{
	BRect	source, dest;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;
	float 	x, y;
	BPoint	p1, p2;

	if (blit_jmp == NULL)
	{
		return -1;
	}


	// initialize drawing

	bounds = BRect(0, 0, 640, 480);

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		// paint background pattern
		SetHighColor (black);
		FillRect (bounds);
		
		SetHighColor (red);
		
		for (x = -bounds.Height(); x < bounds.Width(); x+=20)
		{
			p1.x = x;
			p1.y = 0;
			p2.x = x + bounds.Height() - 1;
			p2.y = bounds.Height() - 1;
			StrokeLine (p1, p2);
		}
		SetHighColor (yellow);
		for (x = 0; x < bounds.Height() + bounds.Width(); x+=20)
		{
			p1.x = x;
			p1.y = 0;
			p2.x = x - (bounds.Height() - 1);
			p2.y = bounds.Height() - 1;
			StrokeLine (p1, p2);
		}
		Sync();
		
		i = 0;
		t1 = system_time();
		while (notstop)
		{
			// Blit at upper part of window
			source.top = bounds.top;
			source.bottom = bounds.top + BLIT_Y;
			dest.top = bounds.top;
			dest.bottom = bounds.top + BLIT_Y;
			for (x = 0; x < bounds.Width() - BLIT_X - 1; x++)
			{
				source.left = bounds.left + x;
				source.right = source.left + BLIT_X;
				dest.left = bounds.left + x + 1;
				dest.right = dest.left + BLIT_X;
				CopyBits (source, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
			}
			
			// blit down at right side
			
			source.left = bounds.right - BLIT_X - 1;
			source.right = bounds.right - 1;
			dest.left = source.left;
			dest.right = source.right;
			
			for (y = 0; y < bounds.Height() - BLIT_Y - 1; y++)
			{
				source.top = bounds.top + y;
				source.bottom = source.top + BLIT_Y;
				dest.top = source.top + 1;
				dest.bottom = dest.top + BLIT_Y;
				CopyBits (source, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
				
			}
			
			// blit left at bottom side
			source.top = bounds.bottom - BLIT_Y - 1;
			source.bottom = bounds.bottom - 1;
			dest.top = bounds.bottom - BLIT_Y - 1;
			dest.bottom = bounds.bottom - 1;
			for (x = bounds.Width() - BLIT_X - 1; x > 0; x--)
			{
				source.left = bounds.left + x;
				source.right = source.left + BLIT_X;
				dest.left = source.left - 1;
				dest.right = dest.left + BLIT_X;
				CopyBits (source, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
			}

			// blit up at left side
			source.left = bounds.left;
			source.right = bounds.left + BLIT_X;
			dest.left = source.left;
			dest.right = source.right;
			
			for (y = bounds.Height() - BLIT_Y - 1; y > 0 ; y--)
			{
				source.top = bounds.top + y;
				source.bottom = source.top + BLIT_Y;
				dest.top = source.top - 1;
				dest.bottom = dest.top + BLIT_Y;
				CopyBits (source, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
				
			}
		}
done:
		Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) BLIT_X * (double) BLIT_Y / (test_time/1e6);
	
	return result;
}

float GfxWindow::RunHlines()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	if ((mDepth == 8) && (line_8 == NULL))
	{
		return -1;
	}
	else if ((mDepth == 32) && (line_32 == NULL))
	{
		return -1;
	}

	bounds = BRect (0, 0, 639, 479);

	// initialize drawing
	SetHighColor (black);
	FillRect (bounds);

	start.x = bounds.left;
	start.y = bounds.Height() / 2;
	end.x = bounds.right;
	end.y = bounds.Height() / 2;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			StrokeLine (start, end);
			Sync();
			if (j++ == runs_10)
			{
				SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		Sync();
		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) bounds.Width() / (test_time/1e6);
	
	return result;
}

float GfxWindow::RunVlines()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	if ((mDepth == 8) && (line_8 == NULL))
	{
		return -1;
	}
	else if ((mDepth == 32) && (line_32 == NULL))
	{
		return -1;
	}

	// initialize drawing
	bounds = BRect (0, 0, 640, 480);

	SetHighColor (black);
	FillRect (bounds);

	start.x = bounds.Width() / 2.0;
	start.y = bounds.left;
	end.x = start.x;
	end.y = bounds.bottom;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			StrokeLine (start, end);
			Sync();
			if (j++ == runs_10)
			{
				SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		Sync();
		t2 = system_time();
		test_time = (t2-t1);
		
		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}


	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) bounds.Height() / (test_time/1e6);
	
	return result;
}

float GfxWindow::RunDlines()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	if ((mDepth == 8) && (line_8 == NULL))
	{
		return -1;
	}
	else if ((mDepth == 32) && (line_32 == NULL))
	{
		return -1;
	}

	bounds = BRect (0, 0, 640, 480);

	// initialize drawing
	SetHighColor (black);
	FillRect (bounds);
	
	start.x = (bounds.Width() - (bounds.Height()-2)) / 2;
	start.y = bounds.Height() / 4;
	end.x = start.x + bounds.Height() - 3;
	end.y = bounds.Height()-2 - bounds.Height() / 4;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			StrokeLine (start, end);
			Sync();
			if (j++ == runs_10)
			{
				SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) (end.x - start.x) / (test_time/1e6);
	
	return result;
}

float GfxWindow::RunHlineArray()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	bounds = BRect (0, 0, 640, 480);

	if ((mDepth == 8) && (line_array_8 == NULL))
	{
		return -1;
	}
	else if ((mDepth == 32) && (line_array_32 == NULL))
	{
		return -1;
	}

	// initialize drawing
	SetHighColor (black);
	FillRect (bounds);
	
	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		c = 0;
		SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			BeginLineArray (bounds.Height());
			start.x = bounds.left;
			end.x = bounds.right;
			for (j = bounds.top; j < bounds.bottom; j++)
			{
				start.y = j;
				end.y = j;
				AddLine (start, end, color_list[(long)(i+j-bounds.left) % 240]);
			}
			EndLineArray();
			Sync();
		}
		Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) bounds.Height() * (double) bounds.Width() / (test_time/1e6);
	
	return result;
}

float GfxWindow::RunVlineArray()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	if ((mDepth == 8) && (line_array_8 == NULL))
	{
		return -1;
	}
	else if ((mDepth == 32) && (line_array_32 == NULL))
	{
		return -1;
	}

	bounds = BRect (0, 0, 640, 480);

	// initialize drawing
	SetHighColor (black);
	FillRect (bounds);

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		c = 0;
		SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			BeginLineArray (bounds.Width());
			start.y = bounds.top;
			end.y = bounds.bottom;
			for (j = bounds.left; j < bounds.right; j++)
			{
				start.x = j;
				end.x = j;
				AddLine (start, end, color_list[(long)(j+i-bounds.left) % 240]);
			}
			EndLineArray();
			Sync();
		}
		Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) bounds.Height() * (double) bounds.Width() / (test_time/1e6);
	
	return result;
}

float GfxWindow::RunDlineArray()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;
	float	xDiff;
	float	xLines;

	if ((mDepth == 8) && (line_array_8 == NULL))
	{
		return -1;
	}
	else if ((mDepth == 32) && (line_array_32 == NULL))
	{
		return -1;
	}

	bounds = BRect (0, 0, 640, 480);

	// initialize drawing
	SetHighColor (black);
	FillRect (bounds);

	start.x = (bounds.Width() - (bounds.Height()-2)) / 2;
	start.y = bounds.Height() / 4;
	end.x = start.x + bounds.Height() - 3;
	end.y = bounds.Height()-2 - bounds.Height() / 4;
	
	xDiff = end.x - start.x;
	xLines = bounds.Width () - xDiff;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			BeginLineArray (xLines);
			for (j = bounds.left; j < bounds.left + bounds.Width() - xDiff; j ++)
			{
				start.x = j;
				end.x = j + xDiff;
				AddLine (start, end, color_list[(long)(i+j-bounds.left) % 240]);
			}
			EndLineArray();
			Sync();
		}
		Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) xLines * (double) xDiff  / (test_time/1e6);
	
	return result;
}

float GfxWindow::RunFillRect()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	if ((mDepth == 8) && (rect_8 == NULL))
	{
		return -1;
	}
	else if ((mDepth == 32) && (rect_32 == NULL))
	{
		return -1;
	}

	// initialize drawing
	bounds = BRect (0,0,640,480);
	SetHighColor (black);
	FillRect (bounds);

	// reduce size by 40 Pixels
	bounds.InsetBy (33,33);
	
	// build random table for movement
	float offsetTable[256][2];
	
	srand ((int) system_time());
	for (j = 0; j < 256; j++)
	{
		offsetTable[j][0] = (float) (rand() % 66);
		offsetTable[j][1] = (float) (rand() % 66);
	}

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			bounds.OffsetTo (offsetTable[i % 256][0], offsetTable[i % 256][1]);

			FillRect (bounds);
			Sync();
			if (j++ == runs_10)
			{
				SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) bounds.Width() * (double) bounds.Height() / (test_time/1e6);
	
	return result;
}
