/*
	
	GfxBench.cpp
	
	core graphic benchmark routines
	
	Copyright 1997 Dirk Steins
	
*/

#ifndef _GFX_BENCH_H
#include "GfxBench.h"
#endif

#include <stdlib.h>

#include "colors.h"
#include "BenchGlobal.h"

GfxBench::GfxBench(BView* view)
		: BObject()
{
	// save view 
	mView = view;
	
	// set sync mode 
	syncMode = SYNC_FLUSH;
	
	// allocate color list
	color_list = (rgb_color*) malloc (sizeof (rgb_color) * MAX_COLS);

	// initialize color cycling table
	int idx = 0;
	int rgbTab[3] = {255, 0, 0};
	int rgbIdx = 0;
	int rgbRunTab[6] = {1, 0, 2, 1, 0, 2};
	int step = 255 / 6;
	for (idx = 0; idx < MAX_COLS; idx++)
	{
		color_list[idx].red = (uchar) rgbTab[0];
		color_list[idx].green =  (uchar) rgbTab[1];
		color_list[idx].blue = (uchar) rgbTab[2];
		color_list[idx].alpha =  (uchar) 0;
		
		rgbTab[rgbRunTab[rgbIdx]] += step;
		if (rgbTab[rgbRunTab[rgbIdx]] > 255)
		{
			rgbTab[rgbRunTab[rgbIdx]] = 255;
			step = -step;
			rgbIdx++;
		}
		else if (rgbTab[rgbRunTab[rgbIdx]] < 0)
		{
			rgbTab[rgbRunTab[rgbIdx]] = 0;
			step = -step;
			rgbIdx++;
		}
		if (rgbIdx > 5) rgbIdx = 0;
	}
}

GfxBench::~GfxBench()
{
	free (color_list);
}

// returns true if test was long enough, 
// otherwise false and calculates new run number
bool GfxBench::CheckRunTime (double test_time, long &runs, long &runs_10)
{
	if (test_time < MIN_GFX_TIME)   /* if total time in test < min time */
	{
		if ((test_time) < MIN_MEASURE)  /* if total time in test < 0.1 */
		{
			runs = MIN_GFX_TIME/MIN_MEASURE* runs; /* increase number of runs by factor 100 */
			runs_10 = runs/240;
		}
		else                         /* if 10 > total time in test > 0.1 */
		{
			runs = MIN_GFX_TIME*MARGINAL/(test_time)* runs; /* increase runs by enough to make tottime > 10 */
			runs_10 = runs/240;
		}
	}
	else                              /* if total time in test > 10 */
	{
		return true;                          /* stop test */
	}                               /* otherwise repeat with new run # */
	return false;
}

void GfxBench::Sync()
{
	switch (syncMode)
	{
		case SYNC_SYNC:
			mView->Sync();
			break;
		case SYNC_FLUSH:
			mView->Flush();
			break;
		default:
			break;
	} 
}

float GfxBench::RunVlines()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;

	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetHighColor (black);
	mView->FillRect (mView->Bounds());

	start.x = mView->Bounds().Width() / 2.0;
	start.y = mView->Bounds().left;
	end.x = start.x;
	end.y = mView->Bounds().bottom;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		mView->SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			mView->StrokeLine (start, end);
			Sync();
			if (j++ == runs_10)
			{
				mView->SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		mView->Sync();
		t2 = system_time();
		test_time = (t2-t1);
		
		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}

//		if (test_time < MIN_GFX_TIME)   /* if total time in test < min time */
//		{
//			if ((test_time) < MIN_MEASURE)  /* if total time in test < 0.1 */
//			{
//				runs = MIN_GFX_TIME/MIN_MEASURE*runs; /* increase number of runs by factor 100 */
//				runs_10 = runs/240;
//			}
//			else                         /* if 10 > total time in test > 0.1 */
//			{
//				runs = MIN_GFX_TIME*MARGINAL/(test_time)*runs; /* increase runs by enough to make tottime > 10 */
//				runs_10 = runs/240;
//			}
//		}
//		else                              /* if total time in test > 10 */
//		{
//			break;                          /* stop test */
//		}                               /* otherwise repeat with new run # */
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) mView->Bounds().Height() / (test_time/1e6);
	
	return result;
}

float GfxBench::RunHlines()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;

	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetHighColor (black);
	mView->FillRect (mView->Bounds());

	start.x = mView->Bounds().left;
	start.y = mView->Bounds().Height() / 2;
	end.x = mView->Bounds().right;
	end.y = mView->Bounds().Height() / 2;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		mView->SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			mView->StrokeLine (start, end);
			Sync();
			if (j++ == runs_10)
			{
				mView->SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		mView->Sync();
		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) mView->Bounds().Width() / (test_time/1e6);
	
	return result;
}

float GfxBench::RunDlines()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetHighColor (black);
	mView->FillRect (mView->Bounds());
	
	bounds = mView->Bounds();

	start.x = (bounds.Width() - (bounds.Height()-2)) / 2;
	start.y = bounds.Height() / 4;
	end.x = start.x + bounds.Height() - 3;
	end.y = bounds.Height()-2 - bounds.Height() / 4;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		mView->SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			mView->StrokeLine (start, end);
			Sync();
			if (j++ == runs_10)
			{
				mView->SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) (end.x - start.x) / (test_time/1e6);
	
	return result;
}

float GfxBench::RunFillRect()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;
	
	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetHighColor (black);
	mView->FillRect (mView->Bounds());
	
	bounds = mView->Bounds();
	
	// reduce size by 40 Pixels
	bounds.InsetBy (33,33);
	
	// build random table for movement
	float offsetTable[256][2];
	
	srand ((int) system_time());
	for (j = 0; j < 256; j++)
	{
		offsetTable[j][0] = (float) (rand() % 66);
		offsetTable[j][1] = (float) (rand() % 66);
	}
	
	runs = RUN_NUM * 15; 		// special start value because of high speed filling
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		mView->SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			bounds.OffsetTo (offsetTable[i % 256][0], offsetTable[i % 256][1]);
			mView->FillRect (bounds);
			Sync();
			if (j++ == runs_10)
			{
				mView->SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height * Width 
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) bounds.Width() * (double) bounds.Height() / ((test_time /* - ((double) runs* 1e3)*/ )/1e6);
	
	return result;
}

float GfxBench::RunPattFill()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;
	pattern	pats[8] = 
			{ 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55,
			0xc7, 0x8f, 0x1f, 0x3e, 0x7c, 0xf8, 0xf1, 0xe3,
			0x1a, 0xa1, 0x1a, 0xa1, 0x1a, 0xa1, 0x1a, 0xa1,
			0xc7, 0x8f, 0x1f, 0x3e, 0x7c, 0xf8, 0xf1, 0xe3,
			0xc7, 0x8f, 0x1f, 0x3e, 0x7c, 0xf8, 0xf1, 0xe3,
			0xc7, 0x8f, 0x1f, 0x3e, 0x7c, 0xf8, 0xf1, 0xe3,
			0xc7, 0x8f, 0x1f, 0x3e, 0x7c, 0xf8, 0xf1, 0xe3,
			0xc7, 0x8f, 0x1f, 0x3e, 0x7c, 0xf8, 0xf1, 0xe3
			};
	pattern*	patPtr;


	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetHighColor (black);
	mView->FillRect (mView->Bounds());
	
	bounds = mView->Bounds();

	// reduce size by 40 Pixels
	bounds.InsetBy (33,33);
	
	// build random table for movement
	float offsetTable[256][2];
	
	srand ((int) system_time());
	for (j = 0; j < 256; j++)
	{
		offsetTable[j][0] = (float) (rand() % 66);
		offsetTable[j][1] = (float) (rand() % 66);
	}

	runs = RUN_NUM;
	runs_10 = runs / 240;
	patPtr = &pats[0];
	while (notstop)
	{
		j = 0;
		c = 0;
		mView->SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			bounds.OffsetTo (offsetTable[i % 256][0], offsetTable[i % 256][1]);
			mView->FillRect (bounds, *patPtr);
			Sync();
			if (j++ == runs_10)
			{
				// patPtr = &pats[c++ % 8];
				mView->SetHighColor (color_list[c++ % 240]);
				j = 0;
			}
		}
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) bounds.Width() * (double) bounds.Height() / (test_time/1e6);
	
	return result;
}

// define size of blit rectangle
float GfxBench::RunBitBlitSS()
{
	BRect	source, dest;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;
	float 	x, y;
	BPoint	p1, p2;


	// initialize drawing

	mView->SetPenSize (1.0);
	
	bounds = mView->Bounds();

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		// paint background pattern
		mView->SetHighColor (black);
		mView->FillRect (mView->Bounds());
		
		mView->SetHighColor (pink);
		for (x = -bounds.Height(); x < bounds.Width(); x+=20)
		{
			p1.x = x;
			p1.y = 0;
			p2.x = x + bounds.Height() - 1;
			p2.y = bounds.Height() - 1;
			mView->StrokeLine (p1, p2);
		}
		mView->SetHighColor (yellow);
		for (x = 0; x < bounds.Height() + bounds.Width(); x+=20)
		{
			p1.x = x;
			p1.y = 0;
			p2.x = x - (bounds.Height() - 1);
			p2.y = bounds.Height() - 1;
			mView->StrokeLine (p1, p2);
		}
		mView->Sync();
		
		i = 0;
		t1 = system_time();
		while (notstop)
		{
			// Blit at upper part of window
			source.top = bounds.top;
			source.bottom = bounds.top + BLIT_Y;
			dest.top = bounds.top;
			dest.bottom = bounds.top + BLIT_Y;
			for (x = 0; x < bounds.Width() - BLIT_X; x++)
			{
				source.left = bounds.left + x;
				source.right = source.left + BLIT_X;
				dest.left = bounds.left + x + 1;
				dest.right = dest.left + BLIT_X;
				mView->CopyBits (source, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
			}
			
			// blit down at right side
			
			source.left = bounds.right - BLIT_X;
			source.right = bounds.right;
			dest.left = source.left;
			dest.right = source.right;
			
			for (y = 0; y < bounds.Height() - BLIT_Y - 1; y++)
			{
				source.top = bounds.top + y;
				source.bottom = source.top + BLIT_Y;
				dest.top = source.top + 1;
				dest.bottom = dest.top + BLIT_Y;
				mView->CopyBits (source, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
				
			}
			
			// blit left at bottom side
			source.top = bounds.bottom - BLIT_Y - 1;
			source.bottom = bounds.bottom - 1;
			dest.top = bounds.bottom - BLIT_Y - 1;
			dest.bottom = bounds.bottom - 1;
			for (x = bounds.Width() - BLIT_X; x > 0; x--)
			{
				source.left = bounds.left + x;
				source.right = source.left + BLIT_X;
				dest.left = source.left - 1;
				dest.right = dest.left + BLIT_X;
				mView->CopyBits (source, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
			}

			// blit up at left side
			source.left = bounds.left;
			source.right = bounds.left + BLIT_X;
			dest.left = source.left;
			dest.right = source.right;
			
			for (y = bounds.Height() - BLIT_Y - 1; y > 0 ; y--)
			{
				source.top = bounds.top + y;
				source.bottom = source.top + BLIT_Y;
				dest.top = source.top - 1;
				dest.bottom = dest.top + BLIT_Y;
				mView->CopyBits (source, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
				
			}
		}
done:	
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) BLIT_X * (double) BLIT_Y / (test_time/1e6);
	
	return result;
}

float GfxBench::RunBitBlitMS()
{
	BBitmap*	bitmap;
	BView*		bView;
	BRect	source, dest;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;
	float 	x, y;
	BPoint	p1, p2;


	// initialize drawing
	mView->SetPenSize (1.0);
	
	bounds = mView->Bounds();
	
	// create Bitmap 
	bitmap = new BBitmap (BRect (0, 0, BLIT_X, BLIT_Y), B_COLOR_8_BIT, true);
	bView = new BView (BRect (0, 0, BLIT_X, BLIT_Y), "BitmapView", B_FOLLOW_NONE, 0);
	bitmap->AddChild (bView);
	
	bitmap->Lock();
	
	// set bitmap pattern
	// paint background pattern
	bView->SetHighColor (black);
	bView->FillRect (bView->Bounds());
	
	bView->SetHighColor (pink);
	for (x = -bounds.Height(); x < bounds.Width(); x+=20)
	{
		p1.x = x;
		p1.y = 0;
		p2.x = x + bounds.Height() - 1;
		p2.y = bounds.Height() - 1;
		bView->StrokeLine (p1, p2);
	}
	bView->SetHighColor (yellow);
	for (x = 0; x < bounds.Height() + bounds.Width(); x+=20)
	{
		p1.x = x;
		p1.y = 0;
		p2.x = x - (bounds.Height() - 1);
		p2.y = bounds.Height() - 1;
		bView->StrokeLine (p1, p2);
	}
	bView->Sync();
	bitmap->Unlock();
	
	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		// paint background pattern
		mView->SetHighColor (black);
		mView->FillRect (mView->Bounds());
		
		mView->SetHighColor (pink);
		for (x = -bounds.Height(); x < bounds.Width(); x+=20)
		{
			p1.x = x;
			p1.y = 0;
			p2.x = x + bounds.Height() - 1;
			p2.y = bounds.Height() - 1;
			mView->StrokeLine (p1, p2);
		}
		mView->SetHighColor (yellow);
		for (x = 0; x < bounds.Height() + bounds.Width(); x+=20)
		{
			p1.x = x;
			p1.y = 0;
			p2.x = x - (bounds.Height() - 1);
			p2.y = bounds.Height() - 1;
			mView->StrokeLine (p1, p2);
		}
		mView->Sync();
		
		i = 0;
		t1 = system_time();
		while (notstop)
		{
			// Blit at upper part of window
			dest.top = bounds.top;
			dest.bottom = bounds.top + BLIT_Y;
			for (x = 0; x < bounds.Width() - BLIT_X; x++)
			{
				dest.left = bounds.left + x;
				dest.right = dest.left + BLIT_X;
				mView->DrawBitmap (bitmap, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
			}
			
			// blit down at right side
			
			dest.left = bounds.right - BLIT_X;
			dest.right = bounds.right;
			
			for (y = 0; y < bounds.Height() - BLIT_Y - 1; y++)
			{
				dest.top = bounds.top + y;
				dest.bottom = dest.top + BLIT_Y;
				mView->DrawBitmap (bitmap, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
			}
			
			// blit left at bottom side
			dest.top = bounds.bottom - BLIT_Y - 1;
			dest.bottom = bounds.bottom - 1;
			for (x = bounds.Width() - BLIT_X; x > 0; x--)
			{
				dest.left = bounds.left + x - 1;
				dest.right = dest.left + BLIT_X;
				mView->DrawBitmap (bitmap, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
			}

			// blit up at left side
			dest.left = bounds.left;
			dest.right = bounds.left + BLIT_X;
			
			for (y = bounds.Height() - BLIT_Y - 1; y > 0 ; y--)
			{
				dest.top = bounds.top + y - 1;
				dest.bottom = dest.top + BLIT_Y;
				mView->DrawBitmap (bitmap, dest);
				Sync();
				i++;
				if (i == runs)
				{
					goto done;
				}
			}
		}
done:			
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// delete allocated bitmap
	delete bitmap;
	
	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = (double) runs * (double) BLIT_X * (double) BLIT_Y / (test_time/1e6);
	
	return result;
}

float GfxBench::RunTextRender()
{
	BPoint	p1;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	char*	string = "AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz0123456789-+!@#$%^&*()";
	BRect	bounds;
	long	xoff, yoff;
	long 	fonth;
	float	dlen;

	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetLowColor (white);
	mView->SetHighColor (white);
	mView->SetDrawingMode (B_OP_COPY);
	mView->FillRect (mView->Bounds());
	mView->SetHighColor (black);
	
	
	// Select font and size
	fonth = 12;
	mView->SetFontName ("Emily");
	mView->SetFontSize (fonth);
	dlen = mView->StringWidth (string);
	bounds = mView->Bounds();
	mView->Sync();

	runs = RUN_NUM;
	xoff = 0;
	yoff = 0;
	while (notstop)
	{
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			p1.x = xoff++ % 29 + 3;
			yoff += fonth + 3;
			p1.y = bounds.Height() - (yoff % (long)(bounds.Height() - 3*fonth) + 2*fonth);
			mView->DrawString (string, p1);
			Sync();
		}
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result =  (double) runs * (double) dlen * (double) fonth / (test_time/1e6);
	
	return result;
}


// Line Array benchmarks
float GfxBench::RunHlineArray()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetHighColor (black);
	mView->FillRect (mView->Bounds());
	
	bounds = mView->Bounds();

	start.x = bounds.Width() / 2.0;
	start.y = bounds.left;
	end.x = start.x;
	end.y = mView->Bounds().bottom;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		c = 0;
		mView->SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			mView->BeginLineArray (bounds.Height());
			start.x = bounds.left;
			end.x = bounds.right;
			for (j = bounds.top; j < bounds.bottom; j++)
			{
				start.y = j;
				end.y = j;
				mView->AddLine (start, end, color_list[(long)(i+j-bounds.left) % 240]);
			}
			mView->EndLineArray();
			Sync();
		}
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result =  (double) runs* (double) bounds.Height() *  (double) bounds.Width() / (test_time/1e6);
	
	return result;
}

float GfxBench::RunVlineArray()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;

	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetHighColor (black);
	mView->FillRect (mView->Bounds());
	
	bounds = mView->Bounds();

	start.x = bounds.Width() / 2.0;
	start.y = bounds.left;
	end.x = start.x;
	end.y = mView->Bounds().bottom;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		c = 0;
		mView->SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			mView->BeginLineArray (bounds.Width());
			start.y = bounds.top;
			end.y = bounds.bottom;
			for (j = bounds.left; j < bounds.right; j++)
			{
				start.x = j;
				end.x = j;
				mView->AddLine (start, end, color_list[(long)(j+i-bounds.left) % 240]);
			}
			mView->EndLineArray();
			Sync();
		}
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result =  (double) runs * (double) bounds.Height() * (double) bounds.Width() / (test_time/1e6);
	
	return result;
}

float GfxBench::RunDlineArray()
{
	BPoint	start, end;
	long	runs, runs_10, i, j, c;
	int		notstop = 1;
	double	t1, t2, test_time;
	BRect	bounds;
	float	xDiff;
	float	xLines;

	// view is attached to window

	// initialize drawing
	mView->SetPenSize (1.0);
	mView->SetHighColor (black);
	mView->FillRect (mView->Bounds());
	
	bounds = mView->Bounds();

	start.x = (bounds.Width() - (bounds.Height()-2)) / 2;
	start.y = bounds.Height() / 4;
	end.x = start.x + bounds.Height() - 3;
	end.y = bounds.Height()-2 - bounds.Height() / 4;
	
	xDiff = end.x - start.x;
	xLines = bounds.Width () - xDiff;

	runs = RUN_NUM;
	runs_10 = runs / 240;
	while (notstop)
	{
		j = 0;
		c = 0;
		mView->SetHighColor (color_list[c]);
		
		t1 = system_time();
		for (i = 0; i < runs; i++)
		{
			mView->BeginLineArray (xLines);
			for (j = bounds.left; j < bounds.left + bounds.Width() - xDiff; j ++)
			{
				start.x = j;
				end.x = j + xDiff;
				mView->AddLine (start, end, color_list[(long)(i+j-bounds.left) % 240]);
			}
			mView->EndLineArray();
			Sync();
		}
		mView->Sync();

		t2 = system_time();
		test_time = (t2-t1);

		if (CheckRunTime (test_time, runs, runs_10))
		{
			break;
		}
	}

	// Pixels = runs * Height
	// Pixels/s = Pixels / test_time in secs
	float result = runs*xLines * xDiff  / (test_time/1e6);
	
	return result;
}
