/*****************************************************************************

	Projet	: Pulsar

	Fichier	:	CFilter.h
	Partie	: Add-ons for filters

	Auteur	: RM
	Date		: 010797 -- version 0.3 (CFilter)
	Format	: tabs==2

	Explanation of add-ons :
	Filters in Pulsar are add-on that draw stuff on the screen based on the
	sound buffers or the real time ffts of the sound.
	Pulsar manages two kind of buffers : 
	...

	This Sample Add-on does nothing. It is just a skeleton that you may use
	for building add-on.

	The simple way to do this :
	- implement & export the C function "CFilter * filterInit(uint32 index)",
	  where  you simply create your CFilter derived class.
	- derive a class from CFilter, you should at least implement two methods
	- CFilter::load() must be implemented, this is where you will fill info about
	  your add-on, name, auhtor, etc. You DON'T need to call the inherited function,
	  it does nothing.
	- CFilter::processFrame() is where you draw the output, by using the info about
	  the sound and the stream.

	You can enhance this usage by implementing the other methods :
	- unload() called just before the add-on image is unloaded, the next call will be
	  the destructor.
	- prepare() is called just before the stream goes on, thus after load() but before
	  processFrame(). You can cache some info, as you may need.
	- terminate() is called when the stream is stopped, thus after any processFrame()
	  but before unload().

	[new to 030797]
	EFilterPosition preferredPosition :
	the filter defines this in order to let the program know that it want to be
	one of first or last called filters. The default value is kFilterPositionAny.
	This is mainly used in the interface rules for guiding the user in the choice of
	its filter order. But, this is not a strict guideline, which means that several
	filters of kind kFilterPositionFirst can be found at the start of the stream; but
	typcally you won't find them after a kFilterPositionAny or kFilterPositionLast.
	Two main rules here :
	- the filters in the list are mandatory in the order kFilterPositionFirst, then
	  kFilterPositionAny then kFilterPositionLast.
	- if the first filter is a kFilterPositionFirst, then this filter HAS the obligation
	  of filling the whole screen or at least erasing it. Which means that in that case
	  the program won't erase the screen, the filter is assumed to do it.
	- if the first filter is NOT a kFilterPositionFirst, the screen will be filled with
	  black.

	[new to 030797 !! the whole mess has been cleanified]
	The structure of the CFilter changes even before I release it !
	There are now sections in it :
	- struct SFilterInfo : info filled by the add-on to let the program has
	  information about him.
	- struct SLoadInfo : info that Pulsar gives to the add-on, this struct
	  is filled between the call to the constructor and the call to the load()
	  method. The add-on can read these, they won't change during its whole life.
	- struct SPrepareInfo : info filled by Pulsar before preare() gets called.
	  These info won't change between the prepare() and the terminate() call, but they
	  might changed between two prepare() call.
	- struct SFrameInfo : info filled by Pulsar just before calling processFrame().
	  This will hopefully change at every frame, which includes the address of the
	  frame buffer or the input sound or fft buffers -- yes, these might change too, since
	  I'm implementing double buffering there...

	The methods always get called like this :
	- filterInit(), called once
	- constructor(), called once
	- [SLoadInfo is filled here]
	- load(), called once
	- loop :
	   - [SPrepareInfo is filled here}
			- prepare()
					- loop : up to 60 or 90 Hz... (machine dependant)
					   - [SFrameInfo is filled here]
					   - several processFrame(), 
			- terminate()
	- unload()
	- destructor()
	- image unloaded.

*****************************************************************************/

#ifndef _H_CFILTER_
#define _H_CFILTER_

#include <OS.h>
#include <image.h>
#include <GameKit.h>
#include <Path.h>

//---------------------------------------------------------------------------
// this is the only function that the add-on must declare
// the function received the index from 0 to n, at it must returns
// a new CFilter derived class if it wants to define a new filter or
// NULL when it wishes to stop.

class CFilter;

typedef CFilter *(filterInitProto) (uint32 index);

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------

//*******************
enum EFilterParamType
//*******************
{
	kFilterParamTypeSingle,
	kFilterParamTypeRange,
	kFilterParamTypeText,
	kFilterParamTypePath
}; // end of EFilterParamType


//*****************
struct SFilterParam
//*****************
{
	SFilterParam(void)	{ name = NULL; text = NULL; path = NULL; }
	~SFilterParam(void)	{ if (name) delete name; if (text) delete text; if (path) delete path; }

	EFilterParamType	type;
	char	*						name;

	double	val,	val2;
	double	def,	def2;
	char		*text;
	BPath		*path;
}; // end of struct defs for SFilterParam

//---------------------------------------------------------------------------


//******************
enum EFilterPosition
//******************
{
	kFilterPositionFirst,		// for those who want to fill up the view
	kFilterPositionAny,			// filters that don't mind (most of them)
	kFilterPositionLast			// for post-video filters, for example bluring...
}; // end of EFilterPosition


//****************
struct SFilterInfo
//****************
{
	// -- info filled by the add-on
	// (NULL at load time)

	char *name;			// name of filter
	char *author;		// author (name, email... one line)
	char *info;			// one line of extra info
	uint32	majorVersion;
	uint32	minorVersion;
	BBitmap	*icon;	// BRect(0,0,31,31) please...

	EFilterPosition preferredPosition;	// defaults to kFilterPositionAny
	color_space preferredMode;	// currently must be B_COLOR_8_BIT
	BList	params;								// a list of pointers on new SFilterParam
	// why not using a BMessage ? --> faster or longer ?

	// -- for the private usage of add-on
	// (NULL at load time - Pulsar won't even look at this,
	//  this is a reminence from C extensions on the Macintosh...)
	void 		*user;

} ; // end of SFilterInfo


//**************
struct SLoadInfo
//**************
{
	// index received by filterInit() when it created this instance
	uint32 index;

	// minor/major version of Pulsar that the add-on can support...
	uint32 majorPulsarVersion;
	uint32 minorPulsarVersion;

	// -- info set by Pulsar
	// (you should not modify nor rely on these)
	bool	enabled;
	bool	running;	
	image_id iid;

	// more usefull stuff -- please read only !
	BDirectory dirApplication;
	BDirectory dirAddon;
	BDirectory dirImg;

}; // end of SLoadInfo


//*****************
struct SPrepareInfo
//*****************
{
	// -- fixed info about play
	// (these are fixed before prepare() gets called)
	int32	sx;
	int32	sy;
	int32 sxy;				// equals sx*sy for fast memsets...
	bool isGameKit;

	color_space mode;	// currently only B_COLOR_8_BIT
	int32	bpr;				// can be != sx in non game kit mode

	BWindowScreen *gameKitScreen;	// only valid if isGameKit is true
}; // end of SPrepareInfo


//***************
struct SFrameInfo
//***************
{
	// -- fixed info about play
	// (these can change at each frame)

	uint8	*screen;

	// -- dynamic info about play
	int32	*rawHigh;
	int32	*rawBass;
	float	*fftHigh;
	float	*fftBass;

	double	averageFps;
	double	lastFps;
	int32		frame;				// frame counter, reset before prepare() gets called

}; // end of SFrameInfo


//***********
class CFilter
//***********
{
public:
	inline CFilter(void);
	virtual inline ~CFilter(void);

	// info filled by the add-on in load() :
	SFilterInfo		sFilter;

	// info filled by Pulsar for the add-on :
	SLoadInfo			sLoad;
	SPrepareInfo	sPrepare;
	SFrameInfo		sFrame;

	// -- callback functions

	virtual bool load(void)						{return false;}
	virtual bool prepare(void)				{return false;}
	virtual void processFrame(void)		{}
	virtual void terminate(void)			{}
	virtual void unload(void)					{}
	virtual void paramChanged(uint32 index)	{} // TBDL -- array ?
	virtual void paramEdited(bool onedit)		{} // TBDL

	// -- some service calls

	// draws a line in the current buffer using bresenham
	void blitLine(long x1, long y1, long x2, long y2, uchar couleur);

	// find a directory by name under the Application directory
	bool findDir(BDirectory *dir, char *name);

	// use datatypes to open the 'name' image in the img directory
	BBitmap *loadBitmap(char *name);

	// use mkimghrd from bedevsupport (aka W. Adams, cf BeNewsLetter 77)
	// and use this to transform the icon from the C header to a BBitmap
	BBitmap * makeIconFromHeader(
						int width, int height,
						color_space cspace,
						int bytesperpixel,
						unsigned char *bits);
	
}; // end of class defs for CFilter



//---------------------------------------------------------------------------
// inline definitions

//***********************************
inline CFilter::CFilter(void)
//***********************************
{
	// -- SFilterInfo stuff

	sFilter.name = NULL;
	sFilter.author = NULL;
	sFilter.info = NULL;
	sFilter.icon = NULL;
	sFilter.majorVersion = 0;
	sFilter.minorVersion = 0;
	sFilter.icon = NULL;

	sFilter.preferredPosition = kFilterPositionAny;
	sFilter.preferredMode = B_COLOR_8_BIT;
	// BList	params; -- empty at first do params.AddItem(new SFilterParam)...

	sFilter.user = NULL;	// here is exception : the only time Pulsar touch this is here.

	// -- SLoadInfo stuff

	sLoad.index = 0;
	sLoad.majorPulsarVersion=0;
	sLoad.minorPulsarVersion=0;

	sLoad.enabled = false;
	sLoad.running = false;
	sLoad.iid = 0;

	// -- SPrepareInfo stuff

	sPrepare.sx = 640;
	sPrepare.sy = 480;
	sPrepare.bpr = 640;
	sPrepare.sxy = sPrepare.bpr*sPrepare.sy;
	sPrepare.isGameKit = false;
	sPrepare.mode = B_COLOR_8_BIT;
	sPrepare.gameKitScreen = NULL;

	// -- SFrameInfo stuff

	sFrame.screen = NULL;
	sFrame.rawHigh = NULL;
	sFrame.rawBass = NULL;
	sFrame.fftHigh = NULL;
	sFrame.fftBass = NULL;
	sFrame.averageFps = 0.0;
	sFrame.lastFps = 0.0;
	sFrame.frame = 0;

} // end of inline constructor for SFilterInfo


//************************************
inline CFilter::~CFilter(void)
//************************************
/*
	Warning : doesn't automagically deletes non-null ptrs.
	I usually do that in my code since I reset deleted ptrs to NULL but I can't
	rely on third-parties to do that in a clean way.
*/
{
	// -- SFilterInfo stuff

	sFilter.name = NULL;
	sFilter.author = NULL;
	sFilter.info = NULL;
	sFilter.icon = NULL;
	sFilter.majorVersion = 0;
	sFilter.minorVersion = 0;
	sFilter.icon = NULL;

	// sFilter.user = NULL; <-- never do that, it's the add-on specific stuff

	// -- SLoadInfo stuff
	
	// no NULL ptrs here...

	// -- SPrepareInfo stuff

	sPrepare.gameKitScreen = NULL;

	// -- SFrameInfo stuff

	sFrame.screen = NULL;
	sFrame.rawHigh = NULL;
	sFrame.rawBass = NULL;
	sFrame.fftHigh = NULL;
	sFrame.fftBass = NULL;

} // end of inline destructor for SFilterInfo

//---------------------------------------------------------------------------


#endif // of _H_CFILTER_

// eoh
