/*
 *  @(#) obuffer.cc 1.15, last edit: 6/23/94 13:04:36
 *  @(#) Copyright (C) 1993, 1994 Tobias Bading (bading@cs.tu-berlin.de)
 *  @(#) Berlin University of Technology
 *
 *  Idea and first implementation for u-law output with fast downsampling by
 *  Jim Boucher (jboucher@flash.bu.edu)
 *
 *  LinuxObuffer class written by
 *  Louis P. Kruger (lpkruger@phoenix.princeton.edu)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *  Changes from version 1.1 to 1.2:
 *    - new LinuxObuffer class, which works with 16 bit soundcards
 *      like Gravis Ultrasound, SoundBlaster 16 or Pro Audio Spectrum 16,
 *      if attached to /dev/dsp
 *    - ShortObuffer renamed to FileObuffer
 *    - If ULAW is defined:
 *      - SparcObuffer feeds mono u-law output to an amd device (u-law format).
 *        The required downsampling to 8 kHz in done in the synthesis filter.
 *      - FileObuffer creates u-law output at 8 kHz instead of 16 bit PCM samples.
 *    - O_NDELAY flag is cleared now after a successful open syscall
 *      on /dev/audio (caused problems under Solaris)
 *    - options -us, -uh and -ul should work now
 *    - FileObuffer can handle incomplete write syscalls now
 */
 
// BeOS version : the BeObuffer class is used to interface a cAudioWrap object
// that handles all incoming buffer sizes

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include "obuffer.h"
#include "header.h"


FileObuffer::FileObuffer (uint32 number_of_channels)
{
#ifdef DEBUG
  if (!number_of_channels || number_of_channels > MAXCHANNELS)
  {
    fprintf(stderr,"FileObuffer: number of channels has to be in [1, %d] !\n",MAXCHANNELS);
    exit (1);
  }
#endif

  channels = number_of_channels;
  for (int i = 0; i < number_of_channels; ++i)
    bufferp[i] = buffer + i;
}


void FileObuffer::append (uint32 channel, int16 value)
{
#ifdef DEBUG
  if (channel >= channels)
  {
    fprintf(stderr,"illegal channelnumber in FileObuffer::append()!\n");
    exit (1);
  }
  if (bufferp[channel] - buffer >= OBUFFERSIZE)
  {
    fprintf(stderr,"FileObuffer: buffer overflow!\n");
    exit (1);
  }
#endif

  *bufferp[channel] = value;
  bufferp[channel] += channels;
}


void FileObuffer::write_buffer (int fd)
{
  int length = (int)((char *)bufferp[0] - (char *)buffer), writelength;

  if ((writelength = write (fd, (char *)buffer, length)) != length)
  {
    // buffer has not been written completely
    if (writelength < 0)
    {
      perror ("write");
      exit (1);
    }
    length -= writelength;
    char *buffer_pos = (char *)buffer;
    do
    {
      buffer_pos += writelength;
      if ((writelength = write (fd, buffer_pos, length)) < 0)
      {
	perror ("write");
	exit (1);
      }
    }
    while (length -= writelength);
  }

  for (int i = 0; i < channels; ++i)
    bufferp[i] = buffer + i;
}


//############################################# BeObuffer ######################

BeObuffer::BeObuffer (uint32 number_of_channels)
{
#ifdef DEBUG
  if (!number_of_channels || number_of_channels > MAXCHANNELS)
  {
    fprintf(stderr,"FileObuffer: number of channels has to be in [1,%d] !\n",MAXCHANNELS);
    exit (1);
  }
#endif

  channels = number_of_channels;
  audio_wrapper = new cAudioWrap(channels);

  for (int i = 0; i < number_of_channels; ++i)
    bufferp[i] = buffer + i;
}

void BeObuffer::append (uint32 channel, int16 value)
{
#ifdef DEBUG
  if (channel >= channels)
  {
    fprintf(stderr,"illegal channelnumber in FileObuffer::append()!\n");
    exit (1);
  }
  if (bufferp[channel] - buffer >= OBUFFERSIZE)
  {
    fprintf(stderr,"FileObuffer: buffer overflow!\n");
    exit (1);
  }
#endif

  *bufferp[channel] = value;
  bufferp[channel] += channels;
}

void BeObuffer::write_buffer (int fd)
{
  long length = (long)((char *)bufferp[0] - (char *)buffer);

  audio_wrapper->AddData((char *)buffer, length);

  for (int i = 0; i < channels; ++i)
    bufferp[i] = buffer + i;
}
