/*************************************************************************

	Project : ZipMe
	Author  : R'alf <mailto:raphael.moll@inforoute.cgs.fr>

	Version : 0.4
	Date    : 20 June 1997

	Format  : tabs == 2	

	Description :
	- The idea is to zip files, cope with it !
	- If only ONE folder or ONE file is selected, the file or the folder name
	  is used to make a "foo.zip" file.
	- If more than one item is selected, the whole stuff is packed under "archive.zip"
	  file name, with an incresing number if entry already exists.

	Based on code from TermHire.cpp, v0.97, release date 24/05/97 by
	Pierre BRUA (brua@dess-info.u-strasbg.fr

	USE AT YOUR OWN RISK !

*************************************************************************/

#include <Directory.h>
#include <Window.h>
#include <StringView.h>
#include <File.h>
#include <Alert.h>
#include <Message.h>
#include <Errors.h>
#include <Path.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

//------------------------------------------------------------------------
// some defines...

#pragma once

// use this define if you want little windows to explain you what's going on
#define VERBOSE

// use this define if you want printf() in the terminal (run a terminal,
// use Olivier's TManager to quit the Tracker, type /boot/system/Tracker & in
// the terminal and you get the stdout here. Too cool !
#undef DPRINTF
//#define DPRINTF

// the name of the add_on
#define ADDON_NAME "ZipMe"
#define ADDON_VERS "0.4"

// change behavior as you want it to. It's a bad idea to remove -r for directory handling !
#define K_ZIP_OPTIONS "-ryq"

#ifndef max
#define max(a,b) ((a)>(b) ? (a) : (b))
#endif

//------------------------------------------------------------------------
// internal state. I use a struct rather than global variables.

struct SState
{
	BWindow *win;
	BTextControl *leaf, *dir, *lasterr, *method, *number;
	BStatusBar *bar;
	int32 errorCount;
	int32 totalCount;
	int32 baseCount;
};

//------------------------------------------------------------------------
// exported function -- the one required by a Tracker Add-on
#pragma export on
void process_refs(entry_ref dir_ref, BMessage* msg, void*);
#pragma export reset

//------------------------------------------------------------------------
// local prototypes 
void error(const char *message);
void createWindow(SState &state);
void closeWindow(SState &state);
void setFileInfo(SState &state, const char *leaf, const char *dir);
void setError(SState &state, const char *error);
void setMethod(SState &state, const char *text);
void setNumber(SState &state, int32 count);

void packDirectory(SState &state, entry_ref &entry);
void packFile(SState &state, entry_ref &ref);
//void packArchive(SState &state, BMessage *msg);



//*****************************
void error(const char *message)
//*****************************
/*
	Displays a message in a dialog box (yes, believe it!)
*/
{
	BAlert *alert = new BAlert(ADDON_NAME " Error :\n", message, "OK");
	alert->Go();
}


//********************************************************************************
void packDirectory(SState &state, entry_ref &ref)
//********************************************************************************
{
BEntry entry(&ref);
BEntry parent;
BPath path;
BPath parentPath;
char buf[B_FILE_NAME_LENGTH*2+128];

	//printf("packDirectory\n");

	entry.GetParent(&parent);
	entry.GetPath(&path);
	parent.GetPath(&parentPath);

	setFileInfo(state, path.Leaf(), parentPath.Path());

	// chdir might not be necessary -- unless I do recursive handling
	// printf("path %s, leaf %s\n", parentPath.Path(), path.Leaf());
	chdir(parentPath.Path());

	sprintf(buf, "zip %s %s.zip %s", K_ZIP_OPTIONS, path.Leaf(), path.Leaf());
	// printf("buf %s\n" ,buf);
	setMethod(state, "zip directory recursively");
	system(buf);
}


//********************************************************************************
void packFile(SState &state, entry_ref &ref)
//********************************************************************************
{
BEntry entry(&ref);
BEntry parent;
BPath path;
BPath parentPath;
char buf[B_FILE_NAME_LENGTH*2+128];

	//printf("packFile\n");

	entry.GetParent(&parent);
	entry.GetPath(&path);
	parent.GetPath(&parentPath);

	setFileInfo(state, path.Leaf(), parentPath.Path());

	// chdir might not be necessary -- unless I do recursive handling
	// printf("path %s, leaf %s\n", parentPath.Path(), path.Leaf());
	chdir(parentPath.Path());

	sprintf(buf, "zip %s %s.zip %s", K_ZIP_OPTIONS, path.Leaf(), path.Leaf());
	// printf("buf %s\n" ,buf);
	setMethod(state, "zip single file");
	system(buf);
}


//********************************************************
void process_refs(entry_ref dir_ref, BMessage* msg, void*)
//********************************************************
/*
	this is the function the Tracker call when the add-on is selected
	dir_ref is the folder you have launched it from
	msg is a BMessage containing the references to BEntry's(Files/Folders)
	and for "void *", I don't know yet...
*/
{
int32 count;	// number of references stored in the BMessage
ulong type;		// type of the BMessage
SState state;
BPath path;

	// be clean and please respect the OS.h header allegory.
	if (!is_computer_on())
	{
		error("Please power-on your computer before running " ADDON_NAME);
		return;
	}

	// might check is_computer_on_fire() to be clean, too

	state.win = NULL;
	state.errorCount = 0;
	state.totalCount = 0;
	state.baseCount = 0;

	// this creates an informative window about the completion of the add-on
	#ifdef VERBOSE
		createWindow(state);
	#endif


	// first, we consider the case when the add-on has been selected
	// from the file menu of the browser window or by right-clicking
	// in the background of the window
	// i.e : in this case, there is no refs on the BMessage
	msg->GetInfo("refs", &type, &count);
	if (count == 0)
	{
		error("Please select a file or a directory to compress !");
	}
	else
	{
		BDirectory dir;
		BEntry entry;

		// if there are datas on the BMessage, but not from the
		// directory type, exit...
		if (dir.SetTo(&dir_ref) != B_NO_ERROR) 
		{
			error("You must launch " ADDON_NAME " from folders.");
			return;
		}
		if(dir.GetEntry(&entry) != B_NO_ERROR)
		{
			error("Sorry, I cannot access the folder while launching " ADDON_NAME);
			return;
		}

		state.totalCount = count;

		#ifdef VERBOSE
		if (state.win && state.bar)
		{
			state.win->Lock();
			state.bar->SetMaxValue(count);
			state.win->Unlock();
		}
		#endif

		try
		{
			for(long i=0; i<count; i++)
			{
				// pack one file or one directory...
				entry_ref ref;
				BDirectory dir;
	
				setNumber(state, i);
				if (msg->FindRef("refs", i, &ref) >= B_NO_ERROR)
				{
					// if the ref is a folder, expand the directory content
					if(dir.SetTo(&ref) >= B_NO_ERROR) packDirectory(state, ref);
					else
					{
						BFile file;
						if(file.SetTo(&ref, B_READ_ONLY) >= B_NO_ERROR) packFile(state, ref);
					}
				}
			}
		}
		catch(...)
		{
			setError(state, "Bad luck !\nException throw !\nProbably IO or File Exception.");
		}
	}
	
	// exit...

	if (state.errorCount > 0)
	{
		char s[256];
		sprintf(s, ADDON_NAME " v." ADDON_VERS " :\n%d error%s detected.",
			state.errorCount, (state.errorCount > 1 ? "s" : ""));
		#ifdef VERBOSE
			if (state.win) state.win->Activate();
		#endif
		error(s);
	}

	#ifdef VERBOSE
		closeWindow(state);
	#endif
}


//******************************
void createWindow(SState &state)
//******************************
{
	BRect area(0,0, 280,80);
	screen_info info;

	get_screen_info(&info);
	area.OffsetBy(info.frame.Width()/2-125, info.frame.Height()/2-80);

	state.win = new BWindow(area, ADDON_NAME " Status", B_TITLED_WINDOW,
										 				B_NOT_ZOOMABLE | B_NOT_V_RESIZABLE | B_NOT_CLOSABLE);
	if (!state.win) return;

	BRect bounds = state.win->Bounds();
	BView *base = new BView(bounds, B_EMPTY_STRING, B_FOLLOW_ALL, B_WILL_DRAW);
	base->SetViewColor(230, 230, 230);
	if (!base) goto error;
	state.win->AddChild(base);

	BRect rect;
	rect.left		= bounds.left  + 10.0;
	rect.top		= bounds.top   + 10.0;
	rect.right	= bounds.right - 10.0;
	rect.bottom	= rect.top+15;

	uint32 rmask = B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP;
	BStringView *str= new BStringView(rect, B_EMPTY_STRING,
														ADDON_NAME " v." ADDON_VERS ", by R'alf", rmask, B_WILL_DRAW);
	if (!str) goto error;	
	base->AddChild(str);

	rect = str->Frame();
	rect.OffsetBy(0.0, rect.Height()+ 20.0);
	state.leaf		= new BTextControl(rect, B_EMPTY_STRING, "File name", "", NULL, rmask);
	if (!state.leaf) goto error;
	base->AddChild(state.leaf);

	rect = state.leaf->Frame();
	rect.OffsetBy(0.0, rect.Height()+ 10.0);
	state.dir			= new BTextControl(rect, B_EMPTY_STRING, "Directory", "", NULL, rmask);
	if (!state.dir) goto error;
	base->AddChild(state.dir);

	rect.OffsetBy(0.0, rect.Height()+ 10.0);
	state.lasterr	= new BTextControl(rect, B_EMPTY_STRING, "Last Error", "", NULL, rmask);
	if (!state.lasterr) goto error;
	base->AddChild(state.lasterr);

	rect.OffsetBy(0.0, rect.Height()+ 10.0);
	state.method	= new BTextControl(rect, B_EMPTY_STRING, "Running", "", NULL, rmask);
	if (!state.method) goto error;
	base->AddChild(state.method);

	state.number = NULL;
	//rect.OffsetBy(0.0, rect.Height()+ 10.0);
	//state.number	= new BTextControl(rect, B_EMPTY_STRING, "File count", "", NULL, rmask);
	//if (!state.number) goto error;
	//base->AddChild(state.number);

	rect.OffsetBy(0.0, rect.Height()+10.0);
	//rect.left += n+2.0;
	state.bar = new BStatusBar(rect, B_EMPTY_STRING, "File count", NULL /*, B_WILL_DRAW, rmask*/);
	if (!state.bar) goto error;
	state.bar->SetFlags(B_WILL_DRAW | B_FULL_UPDATE_ON_RESIZE);
	state.bar->SetResizingMode(rmask);
	base->AddChild(state.bar);
	rect = state.bar->Frame();

	long n=20;
	n = max(state.leaf   ->StringWidth("File name" ), n);
	n = max(state.dir    ->StringWidth("Directory" ), n);
	n = max(state.lasterr->StringWidth("Last Error"), n);
	n = max(state.method ->StringWidth("Running"   ), n);
	//n = max(state.number ->StringWidth("File count"), n);
	state.leaf   ->SetDivider(n + 2.0);
	state.dir    ->SetDivider(n + 2.0);
	state.lasterr->SetDivider(n + 2.0);
	state.method ->SetDivider(n + 2.0);
	//state.number ->SetDivider(n + 2.0);

	state.leaf   ->SetFlags(B_WILL_DRAW);
	state.dir    ->SetFlags(B_WILL_DRAW);
	state.lasterr->SetFlags(B_WILL_DRAW);
	state.method ->SetFlags(B_WILL_DRAW);
	//state.number ->SetFlags(B_WILL_DRAW);


	state.win->ResizeTo(bounds.Width(), rect.bottom+10.0);
	state.win->SetSizeLimits(bounds.Width(), info.frame.Width()-20.0, rect.bottom+10.0, rect.bottom+10.0);
	state.win->Show();
	return;

error:
	state.win = NULL;

} // end of createWindow


//******************************
void 	closeWindow(SState &state)
//******************************
{
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;

	state.win->Lock();
	state.win->Quit();
	int32 val;
	wait_for_thread(state.win->Thread(), &val);

} // end of closeWindow



//****************************************************************
void 	setFileInfo(SState &state, const char *leaf, const char *dir)
//****************************************************************
{
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;
	state.win->Lock();
	if (leaf && state.leaf) state.leaf->SetText(leaf);
	if (dir  && state.dir ) state.dir ->SetText(dir );
	state.win->Unlock();
} // end of setFileInfo


//****************************************************************
void 	setError(SState &state, const char *error)
//****************************************************************
{
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;
	state.win->Lock();
	if (error && state.lasterr) state.lasterr->SetText(error);
	state.win->Unlock();
} // end of setError


//****************************************************************
void 	setMethod(SState &state, const char *text)
//****************************************************************
{
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;
	state.win->Lock();
	if (text && state.method) state.method->SetText(text);
	state.win->Unlock();
} // end of setMethod


//****************************************************************
void 	setNumber(SState &state, int32 count)
//****************************************************************
{
char s[256];
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;
	sprintf(s, "%d of %d", state.baseCount+count, state.totalCount);
	state.win->Lock();
	//if (state.number) state.number->SetText(s);
	if (state.bar) state.bar->Update(1.0, NULL, s);
	state.win->Unlock();
} // end of setNumber


// eoc

