/*************************************************************************

	Project : ExpandMe
	Author  : R'alf <mailto:raphael.moll@inforoute.cgs.fr>

	Version : 0.3
	Date    : 4 June 1997

	Format  : tabs == 2	

	Description :
	- An "expand" add-on for AA-DR9 Tracker.
	- It applies "tar xf" on .tar files, gunzip on .tar?gz and .gz files,
	and "unzip" on .zip files.
	- Recurse througth subdirectories.
	- Expand all selected files.
	- Expand a whole folder if one is selected.
	- Now has some GUI (with as little chrome as possible) for status.

	Based on code from TermHire.cpp, v0.97, release date 24/05/97 by
	Pierre BRUA (brua@dess-info.u-strasbg.fr

	USE AT YOUR OWN RISK !

*************************************************************************/

#include <Directory.h>
#include <Window.h>
#include <StringView.h>
#include <File.h>
#include <Alert.h>
#include <Message.h>
#include <Errors.h>
#include <Path.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

//------------------------------------------------------------------------
// some defines...

#pragma once

// use this define if you want little windows to explain you what's going on
#define VERBOSE

// use this define if you want printf() in the terminal (run a terminal,
// use Olivier's TManager to quit the Tracker, type /boot/system/Tracker & in
// the terminal and you get the stdout here. Too cool !
#undef DPRINTF

// the name of the add_on
#define ADDON_NAME "ExpandMe"
#define ADDON_VERS "0.3"

// the params for the config file
#define EXPANDME_RC			"expandme.rc"
#define EXPANDME_PATH1		"/boot/"
#define EXPANDME_PATH2		"/boot/system/settings/"


#ifndef max
#define max(a,b) ((a)>(b) ? (a) : (b))
#endif

//------------------------------------------------------------------------
// rules list
struct a_rule
{
	char *ext;
	char *command;
};


//------------------------------------------------------------------------
// internal state. I use a struct rather than global variables.

struct SState
{
	BWindow *win;
	BTextControl *leaf, *dir, *lasterr, *method, *number;
	BStatusBar *bar;
	int32 errorCount;
	int32 totalCount;
	int32 baseCount;
	BList	ruleList;		// a list of a_rule ptrs
};

//------------------------------------------------------------------------
// exported function -- the one required by a Tracker Add-on
#pragma export on
void process_refs(entry_ref dir_ref, BMessage* msg, void*);
#pragma export reset

//------------------------------------------------------------------------
// local prototypes 
int expandFile(/*const char *file ,*/ const char *leaf, const char *dir, SState &state);
int expandDirectory(SState &state, BDirectory &dir);
void error(const char *message);
void createWindow(SState &state);
void closeWindow(SState &state);
void setFileInfo(SState &state, const char *leaf, const char *dir);
void setError(SState &state, const char *error);
void setMethod(SState &state, const char *text);
void setNumber(SState &state, int32 count);


//*****************************
void error(const char *message)
//*****************************
/*
	Displays a message in a dialog box (yes, believe it!)
*/
{
	BAlert *alert = new BAlert(ADDON_NAME " Error :\n", message, "OK");
	alert->Go();
}


//********************************************************************************
int expandFile(/*const char * file,*/ const char *leaf, const char *dir, SState &state)
//********************************************************************************
{
char buf[1024];
long len;

	#ifdef DPRINTF
		// remember to uncomment "file" in the args of the function
		printf("expand file %s, leaf %s in dir %s\n", file,leaf,dir);
	#endif
	
	setFileInfo(state, leaf, dir);

	// chdir might not be necessary -- unless I do recursive expand
	chdir(dir);

	len = strlen(leaf);

	// a macro that checks the postfix : 'n' is strlen('x')
	#define POSTFIX(x,n) (len > n && strncmp(&leaf[len-n], x, n) == 0)

	if (POSTFIX(".tar.gz", 7) || POSTFIX(".tgz", 4))
	{
		sprintf(buf, "zcat %s | tar xf -", leaf);
		setMethod(state, "zcat -> untar");
		system(buf);
		
		// don't make it -- why ?
		//long n2 ;
		//if (POSTFIX(".tgz", 4)) n2 = 4; else n2 = 7;
		//sprintf(buf, "tar xf %.*s.tar", leaf, len-n2);
		//printf("Running \"%s\"\n", buf);
		//system(buf);

	}
	else if (POSTFIX(".tar", 4))
	{
		sprintf(buf, "tar xf %s", leaf);
		setMethod(state, "untar");
		system(buf);
	}
	else if (POSTFIX(".gz", 3))
	{
		sprintf(buf, "gunzip %s", leaf);
		setMethod(state, "gunzip");
		system(buf);
	}
	else if (POSTFIX(".zip", 4))
	{
		sprintf(buf, "unzip %s", leaf);
		setMethod(state, "unzip");
		system(buf);
	}
	else if (POSTFIX(".uue", 4))
	{
		sprintf(buf, "uudecode %s", leaf);
		setMethod(state, "uudecode");
		system(buf);
	}
	else
	{
		setError(state, "Can't handle file - wrong type");
		state.errorCount++;
	}

	return B_NO_ERROR;
}



//********************************************************************************
int expandDirectory(SState &state, BDirectory &dir)
//********************************************************************************
{
BDirectory dir2;
int32 count;
status_t status;
BEntry entry;
BFile file;

	status = dir.InitCheck();
	if (status >= B_NO_ERROR) status = dir.Rewind();
	if (status < B_NO_ERROR) return status;

	count = dir.CountEntries();
	if (count < 1) return B_NO_ERROR;
	state.baseCount = state.totalCount;
	state.totalCount += count;

		#ifdef VERBOSE
		if (state.win && state.bar)
		{
			state.win->Lock();
			state.bar->SetMaxValue(state.totalCount);
			state.win->Unlock();
		}
		#endif

	for (int ix=0; ix<count && dir.GetNextEntry(&entry) >= B_NO_ERROR; ix++)
	{
		#ifdef VERBOSE
			setNumber(state, ix);
		#endif

		// if the ref is a folder, expand the directory content
		if(dir2.SetTo(&entry) >= B_NO_ERROR)
		{
			expandDirectory(state, dir2);
		}
		// if the ref is not a folder, launch a Terminal on the parent folder
		else if(entry.InitCheck() >= B_NO_ERROR) 
		{
			// I ask for the parent directory of the entry
			BEntry entry2;
			BPath path, path2;
			entry.GetParent(&entry2);
			entry.GetPath(&path);
			entry2.GetPath(&path2);
			expandFile(/*path.Path(),*/ path.Leaf(), path2.Path(), state);
		}
	}

	return B_NO_ERROR;
}


//********************************************************
void process_refs(entry_ref dir_ref, BMessage* msg, void*)
//********************************************************
/*
	this is the function the Tracker call when the add-on is selected
	dir_ref is the folder you have launched it from
	msg is a BMessage containing the references to BEntry's(Files/Folders)
	and for "void *", I don't know yet...
*/
{
int32 count;	// number of references stored in the BMessage
ulong type;		// type of the BMessage
SState state;
BPath path;

	// be clean and please respect the OS.h header allegory.
	if (!is_computer_on())
	{
		error("Please power-on your computer before running " ADDON_NAME);
		return;
	}

	// might check is_computer_on_fire() to be clean, too

	state.win = NULL;
	state.errorCount = 0;
	state.totalCount = 0;
	state.baseCount = 0;

	// this creates an informative window about the completion of the add-on
	#ifdef VERBOSE
		createWindow(state);
	#endif


	// first, we consider the case when the add-on has been selected
	// from the file menu of the browser window or by right-clicking
	// in the background of the window
	// i.e : in this case, there is no refs on the BMessage
	msg->GetInfo("refs", &type, &count);
	if (count == 0)
	{
		error("Please select a file or a directory to exand !");
	}
	else
	{
		BDirectory dir;
		BEntry entry;

		// if there are datas on the BMessage, but not from the
		// directory type, exit...
		if (dir.SetTo(&dir_ref) != B_NO_ERROR) 
		{
			error("You must launch " ADDON_NAME " from folders.");
			return;
		}
		if(dir.GetEntry(&entry) != B_NO_ERROR)
		{
			error("Sorry, I cannot access the folder while launching " ADDON_NAME);
			return;
		}

		state.totalCount = count;

		#ifdef VERBOSE
		if (state.win && state.bar)
		{
			state.win->Lock();
			state.bar->SetMaxValue(count);
			state.win->Unlock();
		}
		#endif

		entry_ref ref;
		BFile file;
		for (int ix=0; ix<count; ix++)
		{
			if (msg->FindRef("refs", ix, &ref) < B_NO_ERROR) continue;

			#ifdef VERBOSE
				setNumber(state, ix);
			#endif

			// if the ref is a folder, expand the directory content
			if(dir.SetTo(&ref) >= B_NO_ERROR)
			{
				expandDirectory(state, dir);
			}
			// if the ref is not a folder, launch a Terminal on the parent folder
			else if(entry.SetTo(&ref) >= B_NO_ERROR) 
			{
				// I ask for the parent directory of the entry
				BEntry entry2;
				BPath path2;
				entry.GetParent(&entry2);
				entry.GetPath(&path);
				entry2.GetPath(&path2);
				if(expandFile(/*path.Path(),*/ path.Leaf(), path2.Path(), state) != B_NO_ERROR) break;
			}
		}
	}
	
	// exit...

	if (state.errorCount > 0)
	{
		char s[256];
		sprintf(s, ADDON_NAME " v." ADDON_VERS " :\n%d error%s detected.",
			state.errorCount, (state.errorCount > 1 ? "s" : ""));
		#ifdef VERBOSE
			if (state.win) state.win->Activate();
		#endif
		error(s);
	}

	#ifdef VERBOSE
		closeWindow(state);
	#endif
}


//******************************
void createWindow(SState &state)
//******************************
{
	BRect area(0,0, 280,80);
	screen_info info;

	get_screen_info(&info);
	area.OffsetBy(info.frame.Width()/2-125, info.frame.Height()/2-80);

	state.win = new BWindow(area, "ExpandMe Status", B_TITLED_WINDOW,
										 				B_NOT_ZOOMABLE | B_NOT_V_RESIZABLE | B_NOT_CLOSABLE);
	if (!state.win) return;

	BRect bounds = state.win->Bounds();
	BView *base = new BView(bounds, B_EMPTY_STRING, B_FOLLOW_ALL, B_WILL_DRAW);
	base->SetViewColor(230, 230, 230);
	if (!base) goto error;
	state.win->AddChild(base);

	BRect rect;
	rect.left		= bounds.left  + 10.0;
	rect.top		= bounds.top   + 10.0;
	rect.right	= bounds.right - 10.0;
	rect.bottom	= rect.top+15;

	uint32 rmask = B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP;
	BStringView *str= new BStringView(rect, B_EMPTY_STRING,
														ADDON_NAME " v." ADDON_VERS ", by R'alf", rmask, B_WILL_DRAW);
	if (!str) goto error;	
	base->AddChild(str);

	rect = str->Frame();
	rect.OffsetBy(0.0, rect.Height()+ 20.0);
	state.leaf		= new BTextControl(rect, B_EMPTY_STRING, "File name", "", NULL, rmask);
	if (!state.leaf) goto error;
	base->AddChild(state.leaf);

	rect = state.leaf->Frame();
	rect.OffsetBy(0.0, rect.Height()+ 10.0);
	state.dir			= new BTextControl(rect, B_EMPTY_STRING, "Directory", "", NULL, rmask);
	if (!state.dir) goto error;
	base->AddChild(state.dir);

	rect.OffsetBy(0.0, rect.Height()+ 10.0);
	state.lasterr	= new BTextControl(rect, B_EMPTY_STRING, "Last Error", "", NULL, rmask);
	if (!state.lasterr) goto error;
	base->AddChild(state.lasterr);

	rect.OffsetBy(0.0, rect.Height()+ 10.0);
	state.method	= new BTextControl(rect, B_EMPTY_STRING, "Running", "", NULL, rmask);
	if (!state.method) goto error;
	base->AddChild(state.method);

	state.number = NULL;
	//rect.OffsetBy(0.0, rect.Height()+ 10.0);
	//state.number	= new BTextControl(rect, B_EMPTY_STRING, "File count", "", NULL, rmask);
	//if (!state.number) goto error;
	//base->AddChild(state.number);

	rect.OffsetBy(0.0, rect.Height()+10.0);
	//rect.left += n+2.0;
	state.bar = new BStatusBar(rect, B_EMPTY_STRING, "File count", NULL /*, B_WILL_DRAW, rmask*/);
	if (!state.bar) goto error;
	state.bar->SetFlags(B_WILL_DRAW | B_FULL_UPDATE_ON_RESIZE);
	state.bar->SetResizingMode(rmask);
	base->AddChild(state.bar);
	rect = state.bar->Frame();

	long n=20;
	n = max(state.leaf   ->StringWidth("File name" ), n);
	n = max(state.dir    ->StringWidth("Directory" ), n);
	n = max(state.lasterr->StringWidth("Last Error"), n);
	n = max(state.method ->StringWidth("Running"   ), n);
	//n = max(state.number ->StringWidth("File count"), n);
	state.leaf   ->SetDivider(n + 2.0);
	state.dir    ->SetDivider(n + 2.0);
	state.lasterr->SetDivider(n + 2.0);
	state.method ->SetDivider(n + 2.0);
	//state.number ->SetDivider(n + 2.0);

	state.leaf   ->SetFlags(B_WILL_DRAW);
	state.dir    ->SetFlags(B_WILL_DRAW);
	state.lasterr->SetFlags(B_WILL_DRAW);
	state.method ->SetFlags(B_WILL_DRAW);
	//state.number ->SetFlags(B_WILL_DRAW);


	state.win->ResizeTo(bounds.Width(), rect.bottom+10.0);
	state.win->SetSizeLimits(bounds.Width(), info.frame.Width()-20.0, rect.bottom+10.0, rect.bottom+10.0);
	state.win->Show();
	return;

error:
	state.win = NULL;

} // end of createWindow


//******************************
void 	closeWindow(SState &state)
//******************************
{
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;

	state.win->Lock();
	state.win->Quit();
	int32 val;
	wait_for_thread(state.win->Thread(), &val);

} // end of closeWindow



//****************************************************************
void 	setFileInfo(SState &state, const char *leaf, const char *dir)
//****************************************************************
{
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;
	state.win->Lock();
	if (leaf && state.leaf) state.leaf->SetText(leaf);
	if (dir  && state.dir ) state.dir ->SetText(dir );
	state.win->Unlock();
} // end of setFileInfo


//****************************************************************
void 	setError(SState &state, const char *error)
//****************************************************************
{
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;
	state.win->Lock();
	if (error && state.lasterr) state.lasterr->SetText(error);
	state.win->Unlock();
} // end of setError


//****************************************************************
void 	setMethod(SState &state, const char *text)
//****************************************************************
{
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;
	state.win->Lock();
	if (text && state.method) state.method->SetText(text);
	state.win->Unlock();
} // end of setMethod


//****************************************************************
void 	setNumber(SState &state, int32 count)
//****************************************************************
{
char s[256];
#ifndef VERBOSE
	return;
#endif
	if (!state.win) return;
	sprintf(s, "%d of %d", state.baseCount+count, state.totalCount);
	state.win->Lock();
	//if (state.number) state.number->SetText(s);
	if (state.bar) state.bar->Update(1.0, NULL, s);
	state.win->Unlock();
} // end of setNumber


// eoc

