/*
 * AppBar module for Steve Sprang's PowerBar utility.
 */

#include <AppKit.h>
#include <InterfaceKit.h>
#include <StorageKit.h>
#include <stdio.h>
#include <ctype.h>
#include "module.h"

//#define DEBUG

#ifdef DEBUG
#define DPRINTF(x)	printf##x
#else
#define DPRINTF(x)
#endif

/////////// Prototypes /////////////////////

long ReadInAppFile();
void skip_whitespace(char *ptr);
char *get_line(char *buf, FILE *fp);

/////////// Structures /////////////////////

/*
 * These structs are somewhat unnecessary since
 * we don't keep any data around once the menu
 * is built.  AppBar/Win32 uses something like
 * this though, since it can do dynamic reconfiguration
 * of the menu structure and thus needs to keep
 * a separate record of the current menu structure.
 */
typedef struct f {
	char name[30];
	char filename[128];
} app;

typedef struct m {
	char name[30];
	int num_apps;
} menu;

////////////////////////////////////////////

struct MPoint {
	float x, y;
};

// Our diamond symbol

MPoint mp[4] =
{
	{ 15.0, 4.0 },
	{ 26.0, 15.0 },
	{ 15.0, 26.0 },
	{ 4.0, 15.0 }
};

BPoint diamond[4];

//////////////// Globals ///////////////////

BView *owner, *v;
BPopUpMenu *menubar;
BBitmap *bmap;
BRect bounds;
bool dirty;
menu *menu_list;
int line_no = 0;

//////////////// PowerBar functions ////////

long module_get_width()
{
	return kHEIGHT;
}

void module_create(BView *o)
{
	int i;

	owner = o;
	bounds = o->Bounds();

	/*
	 * This is such a ridiculous way
	 * of drawing a polygon.  There must
	 * be a better way but I couldn't figure
	 * it out in 5 minutes...
	 */
	for (i=0;i<4;i++) {
		diamond[i].Set(mp[i].x, mp[i].y);
	}

	bmap = new BBitmap(bounds, B_COLOR_8_BIT, TRUE);
	v = new BView(bounds, "", B_FOLLOW_NONE, B_WILL_DRAW);
	bmap->AddChild(v);
	
	ReadInAppFile();
}

void module_pulse()
{
}

void module_draw()
{
	BPoint tl, tr, bl, br;

	bmap->Lock();
	v->SetHighColor(kBGCOLOR);
	v->FillRect(bounds);

	tl = BPoint(bounds.left, bounds.top);
	tr = BPoint(bounds.right, bounds.top);
	bl = BPoint(bounds.left, bounds.bottom);
	br = BPoint(bounds.right, bounds.bottom);

	v->SetHighColor(kSHADOW);
	v->StrokeLine(br, tr);
	v->StrokeLine(br, bl);

	v->SetHighColor(kHILITE);
	v->StrokeLine(tl, tr);
	v->StrokeLine(tl, bl);

	/*
	 * See note in module_create
	 */
	v->SetHighColor(128,0,128);
	v->FillPolygon(diamond, 4);

	v->Sync();
	bmap->Unlock();

	owner->Window()->Lock();
	owner->DrawBitmapAsync(bmap);
	owner->Window()->Unlock();
}

void module_quit()
{
	delete bmap;
	if (menubar) {
        delete menubar;
    }
}

void module_message_received(BMessage *msg)
{
}

void module_mouse_down(BPoint bp, long buttons)
{
	BMenuItem *item;
	record_ref ref;
	BMessage *bm;
	const char *exe;

	if (!menubar) {
		return;
	} else {
        owner->ConvertToScreen(&bp);
        item = menubar->Go(bp);
        if (item != NULL) {
            bm = item->Message();
            if (bm != NULL) {
                exe = bm->FindString("exe");
                DPRINTF(("Executing %s...\n", exe));
                if (get_ref_for_path(exe, &ref) == B_ERROR) {
                	// File or path does not exist
                    DPRINTF(("Error referencing %s!!\nDoes the file exist?\n",
                    		exe));
                }
                if (be_roster->Launch(ref) != B_NO_ERROR) {
                	DPRINTF(("Error launching!!"));
                }
            }
        }
        module_draw();
    }
}

void module_mouse_moved(BPoint bp, ulong transmit, BMessage *msg)
{
}

/////////////// Special AppBar functions //////////

void skip_whitespace(char *ptr)
{
	while (*ptr == '\n' || *ptr == '\t' || *ptr == ' ') {
		ptr++;
	}
}

/*
 * Reading in text config files is annoyingly tricky.
 * This code was inspired by cfg.c found at www.snippets.org
 */

char *get_line(char *buf, FILE *fp)
{
	char *ptr, *bak;

	while (fgets(buf, 128, fp) != NULL) {
		line_no++;
        ptr = buf;
        skip_whitespace(ptr);
        /*
         * I have no idea why this is needed
         * but sometimes empty lines have 0xA characters
         * in them after skipping the whitespace.
         * Maybe I'm not doing something right.
         */
        if (*ptr == 10) {
        	ptr++;
        }
        /*
         * Case: empty line.
         */
        if (*ptr == '\0') {
            continue;
        }
        /*
         * Case: line with comment at front
         */
        bak = strchr(buf, '#');
        if (ptr == bak) {
            continue;
        }
        /*
         * Trim trailing whitespace or comments
         */
        bak = strrchr(ptr, '#');
        if (bak == NULL) {
            bak = ptr + strlen(ptr);
        }
        while (isspace(*(bak-1))) {
            bak--;
        }
        *bak = '\0';
        return ptr;
    }
    return NULL;
}

long ReadInAppFile()
{
	BMenuItem *cur_bapp;
	BMenu *cur_bmenu;
	BMessage *msg;
	FILE *f;
	int num_menus, num_apps;
	menu cur_menu;
	app cur_app;
	char buf[128];
	char *ptr;

	menubar = NULL;
	f = fopen("/boot/system/settings/appbar.cfg", "r");
	if (f == NULL) {
		BAlert *ba;
		ba = new BAlert("AppBar Module Problem", "/system/settings/appbar.cfg\
				Config File not found.", "Doh...", "Oops...", "Awww...");
		ba->Go();
		return 0;
	}
	ptr = get_line(buf, f);
	if (!ptr) {
        DPRINTF(("Couldn't get a line!\n"));
		goto bad_format;
	}
	num_menus = atoi(ptr);
	if (num_menus == 0) {
        DPRINTF(("No menus?!\n"));
		goto bad_format;
	}
    DPRINTF(("Got %d menus\n", num_menus));
	menubar = new BPopUpMenu("appbar", FALSE, FALSE);
	while (num_menus > 0)
	{
		/*
		 * Get the menu name
		 */
		ptr = get_line(buf, f);
		if (!ptr) {
			goto bad_format;
		}
		strcpy(cur_menu.name, ptr);
        DPRINTF(("Got %s menu\n", cur_menu.name));
        /*
         * Get the number of apps in this menu
         */
        ptr = get_line(buf, f);
        if (!ptr) {
            goto bad_format;
        }
        num_apps = atoi(ptr);
        if (num_apps == 0) {
            goto bad_format;
        }
        DPRINTF(("Got %d apps\n", num_apps));
        /*
         * Fill in the new menu
         */
		cur_bmenu = new BMenu(cur_menu.name);
		while (num_apps > 0) 
		{
			/*
			 * Get app name
			 */
            ptr = get_line(buf, f);
            if (!ptr) {
                goto bad_format;
            }
            strcpy(cur_app.name, ptr);
            DPRINTF(("Got %s app\n", cur_app.name));
            /*
             * Get filename
             */
            ptr = get_line(buf, f);
            if (!ptr) {
                goto bad_format;
            }
            strcpy(cur_app.filename, ptr);
            DPRINTF(("Got %s app filename\n", cur_app.filename));
			msg = new BMessage();
            msg->AddString("exe", cur_app.filename);
            cur_bapp = new BMenuItem(cur_app.name, msg);
            cur_bmenu->AddItem(cur_bapp);
            num_apps--;
        }
		menubar->AddItem(cur_bmenu);
		num_menus--;
	}
	fclose(f);
	return 1;

bad_format:
    BAlert *ba;
    char ms[64];

    if (menubar) {
    	delete menubar;
    }
    sprintf(ms, "Config file problem on line %d\nFix it and restart AppBar.", line_no);
    ba = new BAlert("AppBar Module Problem", ms, "Doh...", "Oops...", "Awww...");
    ba->Go();
    return 0;
}