/*
  rlogin.cpp - rlogin client for BeOS
  Copyright (C) 1998 Uwe Ohse

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.
*/                                                                                            
#include "rlogin.h"

readstdin_c *readstdin;
writestdout_c *writestdout;
static int got_sigterm=0;

// this should be in the stdin reader thread, but keeping it here is easier
// because only one thread needs to care about signals
static struct termios oldmode; 

void myperror(const char *s)
{
    fprintf(stderr, "rlogin: %s: %s\r\n", s, strerror(errno));
}


static void
sigwinch_handler(int sig)
{
}                 

static int 
cleanup(void)
{
	tcsetattr(0,TCSAFLUSH,&oldmode);
}

static void 
sigterm_handler(int sig)
{
	send_signal(readstdin->getid(),SIGTERM);
	send_signal(writestdout->getid(),SIGTERM);
}

static void 
usage(int ecode)
{
	printf("usage: rlogin [options] host\n");
	printf("  options:\n");
	printf("    -8:   eightbit clean (default anyway)\n");
	printf("    -7:   strip bit eight\n");
	printf("    -E:   no escape char\n");
	printf("    -e x: use x as escape char\n");
	printf("    -l u: login as user u\n");
	exit(ecode);
}

static const char *
getspeedstr(struct termios *t)
{
	speed_t s=cfgetispeed(t);
	switch (s) {
	case B0: return "0";
	case B50: return "50";
	case B75: return "75";
	case B110: return "110";
	case B134: return "134";
	case B150: return "150";
	case B200: return "200";
	case B300: return "300";
	case B600: return "600";
	case B1200: return "1200";
	case B2400: return "2400";
	case B4800: return "4800";
	default:
	case B9600: return "9600";
	case B19200: return "19200";
	case B38400: return "38400";
	case B57600: return "57600";
	case B115200: return "115200";
	case B230400: return "230400";
#ifdef B460800
	case B460800: return "460800";
#endif
	}
}



int 
main(int argc, char **argv)
{
	int escapechar='~';
	bool eightbit=true;
	const char *luser=NULL;
	const char *ruser=NULL;
	const char *host=NULL;
	const char *term=NULL;
	struct sockaddr_in sin;
	int port = 0;
	int sock=NULL;
	tcgetattr(0,&oldmode);

	int c;
	while (c=getopt(argc,argv,"l:h78Ee:")) {
		if (c==-1)
			break;
		switch(c)
		{
		default: usage(2); break;
		case 'h': usage(0); break;
		case 'l': ruser=optarg; break;
		case '8': eightbit=true; break;
		case '7': eightbit=false; break;
		case 'E': escapechar=-1; break;
		case 'e':
			{
				char *p;
				int len=strlen(optarg);
				if (len==1)
					escapechar=*optarg;
				else if (*optarg=='\\') {
					// this is stoneage, but compatible.
					escapechar=strtol(optarg+1,NULL,8);
					p=optarg+1+strspn(optarg+1,"01234567");
					if (*p || escapechar<-1 
						|| escapechar>255) {
						fprintf(stderr,
					"rlogind: illegal optarg %s\n",optarg);
						exit(2);
					}
				} else  {
					fprintf(stderr,
					"rlogind: illegal optarg %s\n",optarg);
					exit(2);
				}
			}
			break;
		}
	}
	argc-=optind;
	argv+=optind;
	if (argc>1||argc==0) {
		usage(2);
	}
	host=argv[0];

    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
		myperror("socket");
		exit(1);
    }
    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = INADDR_ANY;
    // unix rlogind wants clients port to be <1024
    // well, windows doesn't restrict total lusers to ports>1023, so this
    // is just annoying, not security (and even under unix there are slowlaris,
    // (s)lackware and other offenders.
    for (port = 1023; port; port--) {
		sin.sin_port = htons(port);
		if (bind(sock, (struct sockaddr *) &sin, sizeof(sin)) >= 0)
			break;
		if (errno != EADDRINUSE) {
			myperror("bind");
			closesocket(sock);
			exit(1);
		}
    }
    if (!port) {
		myperror("all 1023 bind");
		closesocket(sock);
		exit(1);
    }
    sin.sin_port = htons(513); // XXX getservbyname not implemented
    struct hostent *hp=gethostbyname(host);
    if (!hp) {
		sin.sin_addr.s_addr = inet_addr(host);
		if (sin.sin_addr.s_addr==INADDR_NONE) {
			fprintf(stderr,"rlogin: %s: unknown host\n", host);
			exit(1);
		}
		if (connect(sock, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
			myperror("socket");
			exit(1);
		}
	}
    else {
		char **addr=hp->h_addr_list;
		if (!*addr) {
			fprintf(stderr,"rlogin: %s: unknown host\n",
			host);
		}
		while (*addr && addr!=INADDR_ANY) {
			memcpy(&sin.sin_addr.s_addr,*addr,sizeof(*addr));
			if (connect(sock, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
				myperror(*addr);
			} else
				break;
			addr++;
		}
		if (!addr) {
			exit(1);
		}
    }
	luser=getenv("USER");
	if (!luser)
		luser="root";
	if (!ruser)
		ruser=luser;
	{
		char *t2;
		const char *t1=getenv("TERM");
		if (!t1)
			t1="beterm";
		t2=(char *)malloc(strlen(t1)+12);
		if (!t2) {
			fprintf(stderr,"rlogin: out of memory\r\n");
			exit(1);
		}
		sprintf(t2,"%s/%s",t1,getspeedstr(&oldmode));
		term=t2;
	}
    readstdin = new readstdin_c(sock);
    writestdout = new writestdout_c(sock);
    readstdin->escapechar=escapechar;
    readstdin->luser=luser;
    readstdin->ruser=ruser;
    readstdin->term=term;
    readstdin->eightbit=eightbit;

    signal(SIGWINCH, SIG_IGN);
    readstdin->go();
    writestdout->go();
    signal(SIGTERM,sigterm_handler);
    signal(SIGWINCH, SIG_IGN);
    status_t rstatus,status2;
    rstatus=wait_for_thread(writestdout->getid(),&status2);
    if (!got_sigterm) {
	    send_signal(readstdin->getid(),SIGTERM);
    }
    cleanup();
    exit(0);
}
